// ----------------------------------------------------------------------------
// Copyright Zona-OpenGTS
// All rights reserved
// ----------------------------------------------------------------------------
//
// This source module is PROPRIETARY and CONFIDENTIAL.
// NOT INTENDED FOR PUBLIC RELEASE.
// 
// Use of this software is subject to the terms and conditions outlined in
// the 'Commercial' license provided with this software.  If you did not obtain
// a copy of the license with this software please request a copy from the
// Software Provider.
// 
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// ----------------------------------------------------------------------------
// Description:
//  Server Initialization
// ----------------------------------------------------------------------------
// Change History:
//  2014/02/16  Edgar Jose Guinan Ramoni
//     -Initial release
//  2014/02/17  Edgar Jose Guinan Ramoni
//     -This module is to support devices: 
//      Teltonika Series - FM1100, FM2200, FM3100, FM3200, FM3300, FM4100, FM4200, FM5300, AT1000
// ----------------------------------------------------------------------------
package org.opengts.servers.teltonika;

import java.lang.*;
import java.util.*;
import java.io.*;
import java.net.*;
import java.sql.*;

import org.opengts.util.*;
import org.opengts.dbtools.*;
import org.opengts.dbtypes.*;
import org.opengts.db.*;
import org.opengts.db.tables.*;

import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.text.*;
import java.math.BigDecimal;

import org.opengts.servers.*;

public class TrackClientPacketHandler
    extends AbstractClientPacketHandler
{

    public static       boolean DEBUG_MODE                  = false;
    
    // ------------------------------------------------------------------------

    public static       String  UNIQUEID_PREFIX[]           = null;
    public static       double  MINIMUM_SPEED_KPH           = Constants.MINIMUM_SPEED_KPH;
    public static       boolean ESTIMATE_ODOMETER           = true;
    public static       boolean SIMEVENT_GEOZONES           = true;
	public static       boolean SIMEVENT_ENGINEHOURS        = true; // Simulate "engine-hours"
    public static       long    SIMEVENT_DIGITAL_INPUTS     = 0xFFL;
    public static       boolean XLATE_LOCATON_INMOTION      = true;
    public static       double  MINIMUM_MOVED_METERS        = 0.0;
    public static       boolean PACKET_LEN_END_OF_STREAM    = false;
    public static       int     CODEC_GH3000                = 7; // GH12xx/GH3000
    public static       int     CODEC_FMXXXX                = 8;
	
    // ------------------------------------------------------------------------

    /* convenience for converting knots to kilometers */
    public static final double  KILOMETERS_PER_KNOT         = 1.85200000;
	
	/* current device */
    private Device          device                          = null;
    private DataTransport   dataXPort                       = null;
    private String          mobileID                        = null; 
	
	/* duplex/simplex */
    // This value will be set for you by the incoming session to indicate whether
    // the session is TCP (duplex) or UDP (simplex).
    private boolean         isDuplex                        = true;	
	
	/* session IP address */
    // These values will be set for you by the incoming session to indicate the 
    // originating IP address.
    private InetAddress     inetAddress                 = null;
    private String          ipAddress                   = null;
    private int             clientPort                  = 0;
	
    // ------------------------------------------------------------------------

    /* GTS status codes for Input-On events */
    private static final int InputStatusCodes_ON[] = new int[] {
        StatusCodes.STATUS_INPUT_ON_00,
        StatusCodes.STATUS_INPUT_ON_01,
        StatusCodes.STATUS_INPUT_ON_02,
        StatusCodes.STATUS_INPUT_ON_03,
        StatusCodes.STATUS_INPUT_ON_04,
        StatusCodes.STATUS_INPUT_ON_05,
        StatusCodes.STATUS_INPUT_ON_06,
        StatusCodes.STATUS_INPUT_ON_07,
        StatusCodes.STATUS_INPUT_ON_08,
        StatusCodes.STATUS_INPUT_ON_09,
        StatusCodes.STATUS_INPUT_ON_10,
        StatusCodes.STATUS_INPUT_ON_11,
        StatusCodes.STATUS_INPUT_ON_12,
        StatusCodes.STATUS_INPUT_ON_13,
        StatusCodes.STATUS_INPUT_ON_14,
        StatusCodes.STATUS_INPUT_ON_15
    };

    /* GTS status codes for Input-Off events */
    private static final int InputStatusCodes_OFF[] = new int[] {
        StatusCodes.STATUS_INPUT_OFF_00,
        StatusCodes.STATUS_INPUT_OFF_01,
        StatusCodes.STATUS_INPUT_OFF_02,
        StatusCodes.STATUS_INPUT_OFF_03,
        StatusCodes.STATUS_INPUT_OFF_04,
        StatusCodes.STATUS_INPUT_OFF_05,
        StatusCodes.STATUS_INPUT_OFF_06,
        StatusCodes.STATUS_INPUT_OFF_07,
        StatusCodes.STATUS_INPUT_OFF_08,
        StatusCodes.STATUS_INPUT_OFF_09,
        StatusCodes.STATUS_INPUT_OFF_10,
        StatusCodes.STATUS_INPUT_OFF_11,
        StatusCodes.STATUS_INPUT_OFF_12,
        StatusCodes.STATUS_INPUT_OFF_13,
        StatusCodes.STATUS_INPUT_OFF_14,
        StatusCodes.STATUS_INPUT_OFF_15
    };

    // ------------------------------------------------------------------------

    /* GMT/UTC timezone */
    private static final TimeZone gmtTimezone               = DateTime.getGMTTimeZone();

    // ------------------------------------------------------------------------

    /* packet handler constructor */
    public TrackClientPacketHandler() 
    {
        super();
    }

    // ------------------------------------------------------------------------

    /* callback when session is starting */
    public void sessionStarted(InetAddress inetAddr, boolean isTCP, boolean isText)
    {
        super.sessionStarted(inetAddr, isTCP, isText);
        super.clearTerminateSession();
		this.mobileID    = null;
        this.device      = null;
		this.isDuplex    = isTCP;
    }
    
    /* callback when session is terminating */
    public void sessionTerminated(Throwable err, long readCount, long writeCount)
    {
        super.sessionTerminated(err, readCount, writeCount);
    }

    // ------------------------------------------------------------------------
	
	/* returns true if this session is duplex (ie TCP), false if simplex (ie UDP) */
    public boolean isDuplex()
    {
        return this.isDuplex;
    }
	
	// ------------------------------------------------------------------------

    /* based on the supplied packet data, return the remaining bytes to read in the packet */
    public int getActualPacketLength(byte packet[], int packetLen)
    {
        if (PACKET_LEN_END_OF_STREAM) {
            return ServerSocketThread.PACKET_LEN_END_OF_STREAM;
        } else {
            return ServerSocketThread.PACKET_LEN_LINE_TERMINATOR;
        }
    }

    // ------------------------------------------------------------------------

    /* workhorse of the packet handler */	
    public byte[] getHandlePacket(byte pktBytes[]) 
    {	
	   String hex = StringTools.toHexString(pktBytes);
	   String s   = StringTools.toStringValue(pktBytes).trim();
	  // Print.logInfo("Recv[HEX]: " + hex);   
	  // Print.logInfo("Length: " + pktBytes.length);				
       if (ListTools.isEmpty(pktBytes)) {
       Print.logWarn("Ignoring empty/null packet");
       }	   
	   if(isDuplex()) {
       return this.parseInsertRecord_TeltonikaFM_TCP(pktBytes);
	   } else {
       return this.parseInsertRecord_TeltonikaFM_UDP(pktBytes, s);
	   }		
	}

    // ------------------------------------------------------------------------
    // ------------------------------------------------------------------------

    /* Teltonika FMxxxx: parse and insert data record */
    private byte[] parseInsertRecord_TeltonikaFM_TCP(byte pktBytes[])
    {
       
       // AVL Data:
	   // 0000012D1EAC3DBB003FA43A80066C16C0000000C70A00000002000118000001480000010A00
	   // ----------------------------------------------------------------------------
	   // 0000012D1EAC3DBB = (8 Bytes) Timestamp
       // 00               = (1 Bytes) Priority [0 Low, 1 High, 2 Panic, 3 Security]
       /* GPS Element (15 Bytes) */
       // 3FA43A80066C16C0000000C70A0000 
       // 3FA43A80         = (4 Bytes) Longitude 
	   // 066C16C0         = (4 Bytes) Latitude 
	   // 0000             = (2 Bytes) Altitude In meters
	   // 00C7             = (2 Bytes) Angle In degrees
	   // 0A               = (1 Bytes) Satellites Number Of Visible
	   // 0000	           = (2 Bytes) Speed In km/h
	   // 00               = Event 
	   // 02               = IO  
	   // 00               = 1B
	   // 01180000         = 2B
	   // 01480000010A     = 4B 
	   // 00               = 8B
		
		Print.logInfo("Parsing(Teltonika FM): ");
		
         /* Get IMEI */	
	    if (pktBytes.length == 17) {
        mobileID  = StringTools.toStringValue(pktBytes, 2);
     // device  = DCServerFactory.loadDeviceByPrefixedModemID(UNIQUEID_PREFIX, mobileID); 
        device    = DCServerConfig.loadDeviceUniqueID(Main.getServerConfig(), mobileID);		
        if (device == null) {
        Print.logWarn("Unregistered IMEI " + mobileID );
        return "\0".getBytes();
        } else {
        Print.logInfo("Registered IMEI " + mobileID ); 
        return "\1".getBytes(); // Respond 01 If IMEI Accepted
        }}
        if (StringTools.isBlank(mobileID)) {
            Print.logError("'IMEI' value is missing");
            return null;
        }			
        String accountID = device.getAccountID();
        String deviceID  = device.getDeviceID();
        String uniqueID  = device.getUniqueID();    
		
		/* Parsing */
        Payload        p              = new Payload(pktBytes);
        int            tcpHeader      = p.readInt(4,0);
        int            AVLpktLen      = p.readInt(4,0);
        int            CodecID        = p.readInt(1,0);  //08 if FMXXXX
        byte           NumOfRecords   = (byte)p.readInt(1,0);		
		boolean        validGPS       = true;
		long           fixtime        = 0L; 
        int            priority       = 0; // 0 Low, 1 High, 2 Panic, 3 Security      
        double         longitude      = 0.0;
		double         latitude       = 0.0;
        double         altitudeM      = 0.0;
		double         headingDeg     = 0.0;
		int            numSats        = 0;
		double         speedKPH       = 0.0;
        int            EventIOID      = 0;
		int            NumOfIO        = 0;
        int            IOvalue        = 0;
		int            IOindex        = 0;
		double         AIN1           = 0.0; 
		double         AIN2           = 0.0; 
		double         AIN3           = 0.0; 
		double         AIN4           = 0.0; 
        int            statusCode     = StatusCodes.STATUS_LOCATION;
		double         engineHours    = 0.0;
		double         odomKM         = 0.0;
        double         batteryV       = 0.0;
		int            batType; //battery type 12V or 24V
		long           gpioInput      = -1L;	
        int            inputMask      = 0;		
		
        // +++++++++++++++++++++++++++++++++++++++++++++++++++++		
        for (int i = 1; i <= NumOfRecords; i++) {  // Start: For(1) 
            fixtime    = p.readLong(8,1262260800)/1000; // 1/1/2010 00:00:00
            priority   = p.readInt(1,0);
            longitude  = (double)p.readInt(4,0)/10000000;
            latitude   = (double)p.readInt(4,0)/10000000;
            altitudeM  = (double)p.readInt(2,0);
            headingDeg = (double)p.readInt(2,0);
            numSats    = p.readInt(1,0);
            speedKPH   = (double)p.readInt(2,0);
            EventIOID  = p.readInt(1,0);
            NumOfIO    = p.readInt(1,0); //TODO: priority, Event,
			
		/* Alarm Type: Status code */			
		if (EventIOID == 240) {
            statusCode = StatusCodes.STATUS_MOTION_STOP;
        }
		if (EventIOID == 240) {
            statusCode = StatusCodes.STATUS_MOTION_IN_MOTION ;
        }
		if (EventIOID == 69) {
            statusCode = StatusCodes.STATUS_GPS_FAILURE;
        }
		if (EventIOID == 69) {
            //statusCode = StatusCodes.STATUS_CONNECT;
        }
		if (EventIOID == 2) {
            statusCode = StatusCodes.STATUS_ENGINE_STOP;
        }
		if (EventIOID == 2) {
            statusCode = StatusCodes.STATUS_ENGINE_START;
        }
		if (EventIOID == 1){
            statusCode = StatusCodes.STATUS_IGNITION_OFF;
        }
		if (EventIOID == 1) {
            statusCode = StatusCodes.STATUS_IGNITION_ON;
        }
			
		// +++++++++++++++++++++++++++++++++++++++++++++++++++++
        for (int j = 1; j <= 8; j=j*2) { // Start: For(2) 
            NumOfIO = p.readInt(1,0);
            if (NumOfIO >=1) { 
            for (int k = 1; k <= NumOfIO ; k++) {
			IOindex = p.readInt(1,0);
            IOvalue = p.readInt(j,0);			
            switch (IOindex) {
                  case 1:   if (IOvalue >0) inputMask |= 1L    ; break; // DI1
                  case 2:   if (IOvalue >0) inputMask |= 1L <<1; break;
                  case 3:   if (IOvalue >0) inputMask |= 1L <<2; break;
                  case 4:   if (IOvalue >0) inputMask |= 1L <<3; break;
                  case 69:  if (IOvalue >0) inputMask |= 1L <<4; break; // GPSpwr 0:short circuit,1:connected
                  case 240: if (IOvalue >0) inputMask |= 1L <<5; break; // Movement 
                  case 9:   AIN1 = IOvalue;  break; // AIN1, mV
                  case 66:  batteryV = (float)IOvalue/1000;  break; // Power Supply                   				  
			}			
            }}} // End: For(2) 
        // +++++++++++++++++++++++++++++++++++++++++++++++++++++			
		
		gpioInput = (long)inputMask;
		// Battery Type 12V or 24V
		batType = (batteryV<15)? 12 : 24; 
		
        if (fixtime <= 0L) {
            Print.logWarn("Invalid date. ");
            fixtime = DateTime.getCurrentTimeSec(); // default to now
        }    

        /* valid lat/lon? */
        if (validGPS && !GeoPoint.isValid(latitude,longitude)) {
            Print.logWarn("Invalid GPRMC lat/lon: " + latitude + "/" + longitude);
            latitude  = 0.0;
            longitude = 0.0;
            validGPS  = false;
        }
        GeoPoint geoPoint = new GeoPoint(latitude, longitude);

        /* parsed data */
		Print.logInfo("IMEI     : "  + mobileID);
		Print.logInfo("UniqueID  : " + uniqueID);
        Print.logInfo("DeviceID  : " + accountID + "/" + deviceID);        
        Print.logInfo("Timestamp: "  + fixtime + " [" + new DateTime(fixtime) + "]");
        Print.logInfo("GPS      : "  + geoPoint);
        Print.logInfo("Record # : " + NumOfRecords); 
		
        if (statusCode == StatusCodes.STATUS_IGNORE) {
            Print.logError("Ignoring EventCode. ");
            return null;
        } else
        if (statusCode == StatusCodes.STATUS_NONE) {
            statusCode = (speedKPH > 0.0)? StatusCodes.STATUS_MOTION_IN_MOTION : StatusCodes.STATUS_LOCATION;
        } else
        if (XLATE_LOCATON_INMOTION && (statusCode == StatusCodes.STATUS_LOCATION) && (speedKPH > 0.0)) {
            statusCode = StatusCodes.STATUS_MOTION_IN_MOTION;
        }

        /* check IP address */
        DataTransport dataXPort = device.getDataTransport();
        if (this.hasIPAddress() && !dataXPort.isValidIPAddress(this.getIPAddress())) {
            DTIPAddrList validIPAddr = dataXPort.getIpAddressValid(); // may be null
            Print.logError("Invalid IP Address from device: " + this.getIPAddress() + " [expecting " + validIPAddr + "]");
            return null;
        }
        dataXPort.setIpAddressCurrent(this.getIPAddress());    // FLD_ipAddressCurrent
        dataXPort.setRemotePortCurrent(this.getRemotePort());  // FLD_remotePortCurrent
        dataXPort.setLastTotalConnectTime(DateTime.getCurrentTimeSec()); // FLD_lastTotalConnectTime
        if (!dataXPort.getDeviceCode().equalsIgnoreCase(Main.getServerName())) {
            dataXPort.setDeviceCode(Main.getServerName()); // FLD_deviceCode
        }

        /* adjust speed, calculate approximate heading if not available in packet */
        if (speedKPH < MINIMUM_SPEED_KPH) {
            speedKPH   = 0.0;
            headingDeg = 0.0;
        } else
        if (headingDeg < 0.0) {
            headingDeg = 0.0;
            if (validGPS && (device != null)) {
                GeoPoint lastGP = device.getLastValidLocation();
                if (GeoPoint.isValid(lastGP)) {
                    // calculate heading from last point to this point
                    headingDeg = lastGP.headingToPoint(geoPoint);
                }
            }
        }
        Print.logInfo("Speed    : " + StringTools.format(speedKPH,"0.0") + " km/h " + headingDeg);

        /* estimate GPS-based odometer */
        if (odomKM <= 0.0) {
            // calculate odometer
            odomKM = (ESTIMATE_ODOMETER && validGPS)? 
                device.getNextOdometerKM(geoPoint) : 
                device.getLastOdometerKM();
        } else {
            // bounds-check odometer
            odomKM = device.adjustOdometerKM(odomKM);
        }
        Print.logInfo("OdometerKM: " + odomKM);
		
        /* simulate Geozone arrival/departure */
        if (SIMEVENT_GEOZONES && validGPS) {
            java.util.List<Device.GeozoneTransition> zone = device.checkGeozoneTransitions(fixtime, geoPoint);
            if (zone != null) {
                for (Device.GeozoneTransition z : zone) {
                    this.insertEventRecord(device, 
                        z.getTimestamp(), z.getStatusCode(), z.getGeozone(),
                        geoPoint, 0/*gpsAge*/, 0.0/*HDOP*/, numSats,
                        speedKPH, headingDeg, altitudeM, odomKM,
                        gpioInput, batteryV, engineHours);
                    Print.logInfo("Geozone    : " + z);
                    if (z.getStatusCode() == statusCode) {
                        // suppress 'statusCode' event if we just added it here
                        Print.logDebug("StatusCode already inserted: 0x" + StatusCodes.GetHex(statusCode));
                        statusCode = StatusCodes.STATUS_IGNORE;
                    }
                }
            }
        }

        /* status code checks */
        if (statusCode < 0) { // StatusCodes.STATUS_IGNORE
            // skip (event ignored)
        } else
        if (statusCode == StatusCodes.STATUS_IGNORE) {
            // skip (event ignored)
        } else
        if ((statusCode == StatusCodes.STATUS_LOCATION) && this.hasSavedEvents()) {
            // skip (already inserted an event)
        } else
        if (statusCode != StatusCodes.STATUS_LOCATION) {
            this.insertEventRecord(device, 
                fixtime, statusCode, null/*geozone*/,
                geoPoint, 0/*gpsAge*/, 0.0/*HDOP*/, numSats,
                speedKPH, headingDeg, altitudeM, odomKM,
                gpioInput, batteryV, engineHours);
        } else
        if (validGPS && !device.isNearLastValidLocation(geoPoint,MINIMUM_MOVED_METERS)) {
            this.insertEventRecord(device, 
                fixtime, statusCode, null/*geozone*/,
                geoPoint, 0/*gpsAge*/, 0.0/*HDOP*/, numSats,
                speedKPH, headingDeg, altitudeM, odomKM,
                gpioInput, batteryV, engineHours);
        }

        /* save device changes */
        if (!DEBUG_MODE && (device != null)) {
            try {
                //DBConnection.pushShowExecutedSQL();
                device.updateChangedEventFields();
            } catch (DBException dbe) {
                Print.logException("Unable to update Device: " + accountID + "/" + deviceID, dbe);
            } finally {
                //DBConnection.popShowExecutedSQL();
            }
        }
        } // End: For(1)  
        byte[] rtnbytes= {0,0,0,NumOfRecords};
        return rtnbytes;
    }

    // ------------------------------------------------------------------------
    // ------------------------------------------------------------------------
	
	/* Teltonika FMxxxx: parse and insert data record */
    private byte[] parseInsertRecord_TeltonikaFM_UDP(byte pktBytes[],String s) // FM22XX/FM42XX
    {
	
	// only the UDP part
   
        Print.logInfo("Parsing(Teltonika FM): ");

		/* Parsing */
        String   mobileID     = null;
        char[] IMEI;
        int[] intIMEI;
        String hexstring      = new String();
        hexstring             = null;
        String tmpstring      = new String();
        tmpstring             = null;
        IMEI                  = new char[40];
        intIMEI               = new int[40];
        String   headerstr    = "000F";
        String   tcpheaderstr ="00000000";       
        hexstring             = StringTools.toHexString(pktBytes);
        int indexSTARTofIMEI  = hexstring.indexOf("000F") + 4;
        int indexavlpacketID  = hexstring.indexOf("000F") + 2;
        int indexENDofIMEI    = hexstring.indexOf("08",indexSTARTofIMEI - 1);
        int indexAvlData      = indexENDofIMEI+4;
        int indexGPSdata      = indexAvlData+18;
        int i                 = 0;
        int j                 = 0;
  
        mobileID              = hexstring.substring(indexavlpacketID,indexENDofIMEI);
        mobileID              = s.substring(6,22).trim();        
        
        tmpstring             = hexstring.substring(indexAvlData,indexAvlData+16);
        long fixtime          = Long.parseLong(tmpstring,16);
        BigDecimal d_fixtime  = new BigDecimal(fixtime);
        fixtime               = d_fixtime.movePointLeft(3).longValue();

        SimpleDateFormat sdf  = new SimpleDateFormat("MMM dd,yyyy HH:mm");
        java.util.Date resultdate = new java.util.Date(fixtime);  

        tmpstring             = hexstring.substring(indexGPSdata,indexGPSdata+8);
        long longitude_L      = (Long.parseLong(tmpstring,16));
        BigDecimal d_longitude= new BigDecimal(longitude_L);
        double longitude      = d_longitude.movePointLeft(7).doubleValue();
        
        tmpstring             = hexstring.substring(indexGPSdata+8,indexGPSdata+16);
        long latitude_L       = (Long.parseLong(tmpstring,16));
        BigDecimal d_latitude = new BigDecimal(latitude_L);
        double latitude       = d_latitude.movePointLeft(7).doubleValue();       

        tmpstring             = hexstring.substring(indexGPSdata+16,indexGPSdata+20);
        long altitudeM_L      = Long.parseLong(tmpstring,16);
        double   altitudeM    = altitudeM_L;         

        tmpstring             = hexstring.substring(indexGPSdata+26,indexGPSdata+30);
        long speedKPH_L       = Long.parseLong(tmpstring,16);
        double   speedKPH     = speedKPH_L;         
        double   heading      = 0.0;
        int      statusCode   = StatusCodes.STATUS_LOCATION;        
        
        /* no IMEI? */        
        if (StringTools.isBlank(mobileID)) {
            Print.logWarn("MobileID not specified!");
            return null;
        }

        /* GPS Event */
        GPSEvent gpsEvent = new GPSEvent(Main.getServerConfig(), this.ipAddress, this.clientPort, mobileID);
        Device device = gpsEvent.getDevice();
        if (device == null) {
            // errors already displayed
            return null;
        }
		
        gpsEvent.setTimestamp(fixtime);
        gpsEvent.setStatusCode(statusCode);
        gpsEvent.setLatitude(latitude);
        gpsEvent.setLongitude(longitude);
        gpsEvent.setSpeedKPH(speedKPH);
        gpsEvent.setHeading(heading);
        gpsEvent.setAltitude(altitudeM);       

        /* Insert/Return */
        if (this.parseInsertRecord_Common(gpsEvent)) {
		return "0005ABCD01+hexstring.substring(indexSTARTofIMEI,indexavlpacketID+2)+39E".getBytes();         
        } else {
        return ("0005ABCD01"+hexstring.substring(indexavlpacketID,indexavlpacketID+2)+"1E"+"\n").getBytes();
		// return null;
        }
        // return ("0005ABCD01"+hexstring.substring(indexavlpacketID,indexavlpacketID+2)+"1E"+"\n").getBytes();   
    }
	
	// ------------------------------------------------------------------------
    // ------------------------------------------------------------------------
	
	/* parse and insert data record (common) */
    private boolean parseInsertRecord_Common(GPSEvent gpsEvent)
    {
        long fixtime    = gpsEvent.getTimestamp();
        int  statusCode = gpsEvent.getStatusCode();

        /* invalid date? */
        if (fixtime <= 0L) {
            Print.logWarn("Invalid date/time");
            fixtime = DateTime.getCurrentTimeSec(); // default to now
            gpsEvent.setTimestamp(fixtime);
        }
                
        /* valid lat/lon? */
        if (!gpsEvent.isValidGeoPoint()) {
            Print.logWarn("Invalid GPRMC lat/lon: " + gpsEvent.getLatitude() + "/" + gpsEvent.getLongitude());
            gpsEvent.setLatitude(0.0);
            gpsEvent.setLongitude(0.0);
        }
        GeoPoint geoPoint = gpsEvent.getGeoPoint();

        /* minimum speed */
        if (gpsEvent.getSpeedKPH() < MINIMUM_SPEED_KPH) {
            gpsEvent.setSpeedKPH(0.0);
            gpsEvent.setHeading(0.0);
        }

        /* estimate GPS-based odometer */
        Device device = gpsEvent.getDevice();
        double odomKM = 0.0; // set to available odometer from event record
        if (odomKM <= 0.0) {
            odomKM = (ESTIMATE_ODOMETER && geoPoint.isValid())? 
                device.getNextOdometerKM(geoPoint) : 
                device.getLastOdometerKM();
        } else {
            odomKM = device.adjustOdometerKM(odomKM);
        }
        gpsEvent.setOdometerKM(odomKM); 
		
        /* create/insert standard event */
        gpsEvent.insertEventData(fixtime, statusCode);
        /* save device changes */
        gpsEvent.updateDevice();
        /* return success */
        return true;
    }
	
	// ------------------------------------------------------------------------
    // ------------------------------------------------------------------------

    private EventData createEventRecord(Device device, 
        long     gpsTime, int statusCode, Geozone geozone,
        GeoPoint geoPoint, long gpsAge, double HDOP, int numSats,
        double   speedKPH, double heading, double altitudeM, double odomKM,
        long     gpioInput, double batteryV, double engineHours)
    {
        String accountID    = device.getAccountID();
        String deviceID     = device.getDeviceID();
        EventData.Key evKey = new EventData.Key(accountID, deviceID, gpsTime, statusCode);
        EventData evdb      = evKey.getDBRecord();
        evdb.setGeozone(geozone);
        evdb.setGeoPoint(geoPoint);
        evdb.setGpsAge(gpsAge);
        evdb.setHDOP(HDOP);
        evdb.setSatelliteCount(numSats);
        evdb.setSpeedKPH(speedKPH);
        evdb.setHeading(heading);
        evdb.setAltitude(altitudeM);
        evdb.setOdometerKM(odomKM);
        evdb.setInputMask(gpioInput);
        evdb.setBatteryVolts(batteryV);
		evdb.setEngineHours(engineHours);
        return evdb;
    }

    /* create and insert an event record */
    private void insertEventRecord(Device device, 
        long     gpsTime, int statusCode, Geozone geozone,
        GeoPoint geoPoint, long gpsAge, double HDOP, int numSats,
        double   speedKPH, double heading, double altitudeM, double odomKM,
        long     gpioInput, double batteryV, double engineHours)
    {

        /* create event */
        EventData evdb = createEventRecord(device, 
            gpsTime, statusCode, geozone,
            geoPoint, gpsAge, HDOP, numSats,
            speedKPH, heading, altitudeM, odomKM,
            gpioInput, batteryV, engineHours);

        /* insert event */
        // this will display an error if it was unable to store the event
        Print.logInfo("Event: [0x" + StringTools.toHexString(statusCode,16) + "] " + 
            StatusCodes.GetDescription(statusCode,null));
        if (!DEBUG_MODE) {
            device.insertEventData(evdb);
            this.incrementSavedEventCount();
        }

    }

    // ------------------------------------------------------------------------
    // ------------------------------------------------------------------------
    // ------------------------------------------------------------------------

    public static void configInit() 
    {
        DCServerConfig dcsc = Main.getServerConfig();
        if (dcsc != null) {
    
            /* common */
            UNIQUEID_PREFIX          = dcsc.getUniquePrefix();
            MINIMUM_SPEED_KPH        = dcsc.getMinimumSpeedKPH(MINIMUM_SPEED_KPH);
            ESTIMATE_ODOMETER        = dcsc.getEstimateOdometer(ESTIMATE_ODOMETER);
            SIMEVENT_GEOZONES        = dcsc.getSimulateGeozones(SIMEVENT_GEOZONES);
            SIMEVENT_DIGITAL_INPUTS  = dcsc.getSimulateDigitalInputs(SIMEVENT_DIGITAL_INPUTS) & 0xFFL;
            XLATE_LOCATON_INMOTION   = dcsc.getStatusLocationInMotion(XLATE_LOCATON_INMOTION);
            MINIMUM_MOVED_METERS     = dcsc.getMinimumMovedMeters(MINIMUM_MOVED_METERS);

            /* custom */
            PACKET_LEN_END_OF_STREAM = dcsc.getBooleanProperty(Constants.CFG_packetLenEndOfStream, PACKET_LEN_END_OF_STREAM);

        }
        
    }

}
