package org.opengts.servers.meitrack;

import java.util.*;
import java.io.*;
import java.net.*;
import java.nio.*;

import javax.swing.JOptionPane;

import org.opengts.util.*;
import org.opengts.db.*;
import org.opengts.db.tables.*;

import org.opengts.servers.*;

/**
*** <code>TrackClientPacketHandler</code> - This module contains the general
*** "business logic" for parsing incoming data packets from the remote tracking
*** Device.
**/

public class TrackClientPacketHandler
    extends AbstractClientPacketHandler
{
    public  static 		int	DATA_FORMAT_OPTION				= 1;

    //    vehicle value.
    public  static		boolean ESTIMATE_ODOMETER          = true;
    
    /* simulate geozone arrival/departure */
    // (enable to insert simulated Geozone arrival/departure EventData records)
    public  static       boolean SIMEVENT_GEOZONES          = true;
    
    /* simulate digital input changes */
    public  static       long    SIMEVENT_DIGITAL_INPUTS    = 0x0000L; // 0xFFFFL;

    /* flag indicating whether data should be inserted into the DB */
    // should be set to 'true' for production.
    private static       boolean DFT_INSERT_EVENT           = true;
    private static       boolean INSERT_EVENT               = DFT_INSERT_EVENT;

    /* update Device record */
    // (enable to update Device record with current IP address and last connect time)
  //private static       boolean UPDATE_DEVICE              = false;
    
    /* minimum acceptable speed value */
    // Speeds below this value should be considered 'stopped'
    public  static       double  MINIMUM_SPEED_KPH          = 0.0;

    /* Knot/Kilometer conversions */
    public static final double  KILOMETERS_PER_KNOT     = 1.85200000;
    public static final double  KNOTS_PER_KILOMETER     = 1.0 / KILOMETERS_PER_KNOT;

    // ------------------------------------------------------------------------

    /* Ingore $GPRMC checksum? */
    // (only applicable for data formats that include NMEA-0183 formatted event records)
    //private static       boolean IGNORE_NMEA_CHECKSUM       = false;

    // ------------------------------------------------------------------------

    /* GMT/UTC timezone */
    private static final TimeZone gmtTimezone               = DateTime.getGMTTimeZone();

    // ------------------------------------------------------------------------
    
    private 			String 		tipoData				=	"";

    /* GTS status codes for Input-On events */
    private static final int InputStatusCodes_ON[] = new int[] {
        StatusCodes.STATUS_INPUT_ON_00,
        StatusCodes.STATUS_INPUT_ON_01,
        StatusCodes.STATUS_INPUT_ON_02,
        StatusCodes.STATUS_INPUT_ON_03,
        StatusCodes.STATUS_INPUT_ON_04,
        StatusCodes.STATUS_INPUT_ON_05,
        StatusCodes.STATUS_INPUT_ON_06,
        StatusCodes.STATUS_INPUT_ON_07,
        StatusCodes.STATUS_INPUT_ON_08,
        StatusCodes.STATUS_INPUT_ON_09,
        StatusCodes.STATUS_INPUT_ON_10,
        StatusCodes.STATUS_INPUT_ON_11,
        StatusCodes.STATUS_INPUT_ON_12,
        StatusCodes.STATUS_INPUT_ON_13,
        StatusCodes.STATUS_INPUT_ON_14,
        StatusCodes.STATUS_INPUT_ON_15
    };

    /* GTS status codes for Input-Off events */
    private static final int InputStatusCodes_OFF[] = new int[] {
        StatusCodes.STATUS_INPUT_OFF_00,
        StatusCodes.STATUS_INPUT_OFF_01,
        StatusCodes.STATUS_INPUT_OFF_02,
        StatusCodes.STATUS_INPUT_OFF_03,
        StatusCodes.STATUS_INPUT_OFF_04,
        StatusCodes.STATUS_INPUT_OFF_05,
        StatusCodes.STATUS_INPUT_OFF_06,
        StatusCodes.STATUS_INPUT_OFF_07,
        StatusCodes.STATUS_INPUT_OFF_08,
        StatusCodes.STATUS_INPUT_OFF_09,
        StatusCodes.STATUS_INPUT_OFF_10,
        StatusCodes.STATUS_INPUT_OFF_11,
        StatusCodes.STATUS_INPUT_OFF_12,
        StatusCodes.STATUS_INPUT_OFF_13,
        StatusCodes.STATUS_INPUT_OFF_14,
        StatusCodes.STATUS_INPUT_OFF_15
    };

    // ------------------------------------------------------------------------
    // ------------------------------------------------------------------------

    /* TCP session ID */
    private String          sessionID                   = null;

    /* common GPSEvent instance */
    private GPSEvent        gpsEvent                    = null;

    /* Device record */
    private Device          gpsDevice                   = null;
    private String          lastModemID                 = null;

    /* Session 'terminate' indicator */
    // This value should be set to 'true' when this server has determined that the
    // session should be terminated.  For instance, if this server finishes communication
    // with the Device or if parser finds a fatal error in the incoming data stream 
    // (ie. invalid Account/Device, or unrecognizable data).
    private boolean         terminate                   = false;

    /* session IP address */
    // These values will be set for you by the incoming session to indicate the 
    // originating IP address.
    private String          ipAddress                   = null;
    private int             clientPort                  = 0;

    /* packet handler constructor */
    public TrackClientPacketHandler() 
    {
        super();
        //Print.logStackTrace("new TrackClientPacketHandler ...");
    }

    // ------------------------------------------------------------------------

    /* callback when session is starting */
    // this method is called at the beginning of a communication session
    public void sessionStarted(InetAddress inetAddr, boolean isTCP, boolean isText)
    {
        super.sessionStarted(inetAddr, isTCP, isText);
        super.clearTerminateSession();

        /* init */
        this.ipAddress        = (inetAddr != null)? inetAddr.getHostAddress() : null;
        this.clientPort       = this.getSessionInfo().getRemotePort();
        if (this.isDuplex()) {
        	this.tipoData="TCP";
            //Print.logInfo("Inicio TCP: " + this.ipAddress + " [" + new DateTime() + "]");
        	//	Print.logInfo("========================================================");
        } else {
        	this.tipoData="UDP";
        	//Print.logInfo("Inicio UDP: " + this.ipAddress + " [" + new DateTime() + "]");
        	//Print.logInfo("========================================================");
        }

    }

    /* callback when session is terminating */
    // this method is called at the end of a communication session
    public void sessionTerminated(Throwable err, long readCount, long writeCount)
    {
        super.sessionTerminated(err, readCount, writeCount);
    }

    /* callback to return the TCP session id */
    public String getSessionID()
    {
        if (!StringTools.isBlank(this.sessionID)) {
            return this.sessionID;
        } else
        if (this.gpsDevice != null) {
            return CreateTcpSessionID(this.gpsDevice);
        } else
        if (this.gpsEvent != null) {
            return CreateTcpSessionID(this.gpsEvent.getDevice());
        } else {
            return null;
        }
    }

    // ------------------------------------------------------------------------

    public static final boolean USE_STANDARD_TCP_SESSION_ID = true;

    /* get TCP session ID */
    public static String GetTcpSessionID(Device dev)
    {
        if (USE_STANDARD_TCP_SESSION_ID || (dev == null)) {
            return DCServerFactory.getTcpSessionID(dev);
        } else {
            // Custom example for extracting the session ID from the Device 
            // UniqueID (instead of the DeviceID)
            String uid = dev.getUniqueID();
            int    u   = uid.lastIndexOf("_");
            if (u < 0) {
                // no "_" character, fallback to standard TCP sessionID
                return DCServerFactory.getTcpSessionID(dev);
            } else {
                String sessID = dev.getAccountID() + "/" + uid.substring(0,u+1);
                return sessID;
            }
        }
    }

    /* create TCP session ID */
    public static String CreateTcpSessionID(Device dev)
    {
        if (USE_STANDARD_TCP_SESSION_ID || (dev == null)) {
            return DCServerFactory.createTcpSessionID(dev);
        } else {
            String uid = dev.getUniqueID();
            int    u   = uid.lastIndexOf("_");
            if (u < 0) {
                // no "_" character, fallback to standard TCP sessionID
                return DCServerFactory.createTcpSessionID(dev);
            } else {
                String sessID = dev.getAccountID() + "/" + uid.substring(0,u+1);
                dev.setLastTcpSessionID(sessID);
                return sessID;
            }
        }
    }

     /**
     *** Create GPSEvent from the ModemID
     *** @return The created GPSEvent instance, or null if the ModemID is invalid
     **/
     private GPSEvent createGPSEvent(String modemID)
     {
         DCServerConfig dcserver = Main.getServerConfig(null);

         /* create/load device */
         if (this.gpsDevice != null) {
             if (!modemID.equals("*") && 
                 ((this.lastModemID == null) || !this.lastModemID.equals(modemID))) { // fix [B10]
                 Print.logError("New MobileID does not match previously loaded Device");
                 //return this.gpsEvent;
                 return null;
             }
             this.gpsEvent  = new GPSEvent(dcserver, this.ipAddress, this.clientPort, this.gpsDevice);
         } else {
             if (StringTools.isBlank(modemID) || modemID.equals("*")) {
                 Print.logWarn("ModemID not specified!");
                 return null;
             }
             this.gpsEvent  = new GPSEvent(dcserver, this.ipAddress, this.clientPort, modemID);
             this.gpsDevice = this.gpsEvent.getDevice(); // may still be null
         }

         /* no Device? */
         if (this.gpsDevice == null) {
             // errors already displayed
             return null;
         }

         /* create session ID */
         this.sessionID = CreateTcpSessionID(this.gpsDevice);

         /* return GPSEvent */
         return this.gpsEvent; // non-null

     }
     
     /**
      * ***********************************************************************
      *  Compilation:  javac CRC16CCITT.java
      *  imatveev13
      *  www.cs.princeton.edu/introcs/51data/CRC16CCITT.java.html
      *  Execution:    java CRC16CCITT s
      *  Dependencies: 
      *  
      *  Reads in a sequence of bytes and prints out its 16 bit
      *  Cylcic Redundancy Check (CRC-CCIIT 0xFFFF).
      *
      *  1 + x + x^5 + x^12 + x^16 is irreducible polynomial.
      *
      *  % java CRC16-CCITT 123456789
      *  CRC16-CCITT = 29b1
      *
      ************************************************************************
      **/
     public static int CRC16CCITT(byte[] bytes) {
     	int crc = 0xFFFF;          // initial value
     	int polynomial = 0x1021;   // 0001 0000 0010 0001  (0, 5, 12) 

     	for (byte b : bytes) {
     		for (int i = 0; i < 8; i++) {
     			boolean bit = ((b   >> (7-i) & 1) == 1);
     			boolean c15 = ((crc >> 15    & 1) == 1);
     			crc <<= 1;
     			if (c15 ^ bit) crc ^= polynomial;
     		}
     	}

     	crc &= 0xffff;
     	//System.out.println("CRC16-CCITT = " + Integer.toHexString(crc));
     	return crc;
     }

     byte[] packetGetDeviceIdBA(byte[] packet){
     	byte[] devId = new byte[7]; 
     	System.arraycopy(packet, 4, devId, 0, 7);
     	return devId;
     }
     
     String packetGetDeviceIdStr(byte[] packet){
     	
     	byte[] devID_ba = new byte[7]; 
     	System.arraycopy(packet, 4, devID_ba, 0, 7);

     	String devIdStr = StringTools.toHexString(devID_ba);
     	Print.logInfo("devIdStr: " + devIdStr);

         //remove F padding; see meitrack manual on device ID format
     	devIdStr = devIdStr.toLowerCase();
     	while( devIdStr.endsWith("f")){
     		devIdStr = devIdStr.substring(0, devIdStr.length()-1);
     	}

     	return devIdStr;
     }

     int packetGetLength(byte[] packet){
    	 Payload p = new Payload(packet, 2,2);
    	 int p_int	= p.readUInt(90, 0);
    	 Print.logInfo("packet length: " + p_int);
    	 //return(length);
    	 return(p_int);
     }

     int packetGetDeviceCommand(byte[] packet){
    	 
    	 Payload device_cmd_p = new Payload(packet, 11, 2);
    	 int device_cmd = device_cmd_p.readUInt(2,0);
    	 System.out.println("Comando: " + Integer.toHexString(device_cmd));
    	 
    	 return device_cmd;
     }
     
     //the packet shell be with '\r\n'
     boolean packetCheckCRC(byte[] packet){
     	int length = packet.length;
     	if(length < Constants.MIN_PACKET_LENGTH){
     		return(false);
     	}
     	
     	Payload crc_pkt_p = new Payload(packet, length - 4, 2);
     	int crc_pkt = crc_pkt_p.readUInt(2,0); // readLong(7,0L);
     	
     	byte[] packet_no_crc = new byte[length-4];
     	System.arraycopy(packet, 0, packet_no_crc, 0, length - 4);
     	int crc_calc = CRC16CCITT(packet_no_crc);
     	
     	if(crc_pkt != crc_calc){
     		Print.logInfo("BAD crc");
     		Print.logInfo("crc_calc: " + crc_calc + " crc_pkt:" + crc_pkt);
     		return(false);
     	}
     	Print.logInfo("GOOD crc");
     	return(true);
     }
     
     byte[] packetDeviceLoginAccept(byte[] packet){
     	//
     	//@@lliiiiiiiccssrn
     	//hex:40400011085716520874FF4000A2670D0A
     	//TODO:rewrite the function to use Payload.write
     	int length = 17;//length of device login accept packet
     	
     	ByteBuffer packet_bb = ByteBuffer.allocate(length);
     	
     	packet_bb.put((byte)0x40);//from server to device '@' 
     	packet_bb.put((byte)0x40);//from server to device '@'
     	packet_bb.putChar((char)length);//packet length
     	packet_bb.put(packet, 4, 7);//device id
     	packet_bb.putChar((char)0x4000);//command:login acceped
     	int crc = CRC16CCITT(packet_bb.array());
     	packet_bb.putChar((char)crc);
     	packet_bb.put((byte)0x0D);//'\r'
     	packet_bb.put((byte)0x0A);//'\n'
     	Print.logInfo("packet_bb[HEX]: " + StringTools.toHexString(packet_bb.array()));
     	return(packet_bb.array());
     }
     
     /**
      * Calcula los segundos en EpochTime de UTC
      **/
     private long _getUTCSeconds(long dmy, long hms){
    	 //Print.logInfo("dmy : " + dmy + " hms : " + hms);
     
         /* time of day [TOD] */
         int    HH  = (int)((hms / 10000L) % 100L);
         int    MM  = (int)((hms / 100L) % 100L);
         int    SS  = (int)(hms % 100L);
         long   TOD = (HH * 3600L) + (MM * 60L) + SS;
     
         /* current UTC day */
         long DAY;
         if (dmy > 0L) {
             int    yy  = (int)(dmy % 100L) + 2000;
             int    mm  = (int)((dmy / 100L) % 100L);
             int    dd  = (int)((dmy / 10000L) % 100L);
             long   yr  = ((long)yy * 1000L) + (long)(((mm - 3) * 1000) / 12);
             DAY        = ((367L * yr + 625L) / 1000L) - (2L * (yr / 1000L))
                          + (yr / 4000L) - (yr / 100000L) + (yr / 400000L)
                          + (long)dd - 719469L;
         } else {
             // we don't have the day, so we need to figure out as close as we can what it should be.
             long   utc = DateTime.getCurrentTimeSec();
             long   tod = utc % DateTime.DaySeconds(1);
             DAY        = utc / DateTime.DaySeconds(1);
             long   dif = (tod >= TOD)? (tod - TOD) : (TOD - tod); // difference should be small (ie. < 1 hour)
             if (dif > DateTime.HourSeconds(12)) { // 12 to 18 hours
                 // > 12 hour difference, assume we've crossed a day boundary
                 if (tod > TOD) {
                     // tod > TOD likely represents the next day
                     DAY++;
                 } else {
                     // tod < TOD likely represents the previous day
                     DAY--;
                 }
             }
         }
         
         /* return UTC seconds */
         long sec = DateTime.DaySeconds(DAY) + TOD;
         return sec;
         
     }
     
     /**
      *** Parses latitude given values from GPS device.
      *** @param  s  Latitude String from GPS device in ddmm.mm format.
      *** @param  d  Latitude hemisphere, "N" for northern, "S" for southern.
      *** @return Latitude parsed from GPS data, with appropriate sign based on hemisphere or
      ***         90.0 if invalid latitude provided.
      **/
     private double _parseLatitude(String s, String d)
     {
    	 double _lat = StringTools.parseDouble(s, 99999.0);
    	 if (_lat < 99999.0) {
    		 double lat = (double)((long)_lat / 100L); // _lat is always positive here
    		 lat += (_lat - (lat * 100.0)) / 60.0;
    		 return d.equals("S")? -lat : lat;
    	 } else {
    		 return 90.0; // invalid latitude
    	 }
     }

     /**
      *** Parses longitude given values from GPS device.
      *** @param s Longitude String from GPS device in ddmm.mm format.
      *** @param d Longitude hemisphere, "E" for eastern, "W" for western.
      *** @return Longitude parsed from GPS data, with appropriate sign based on hemisphere or
      *** 180.0 if invalid longitude provided.
      **/
     private double _parseLongitude(String s, String d)
     {
    	 double _lon = StringTools.parseDouble(s, 99999.0);
    	 if (_lon < 99999.0) {
    		 double lon = (double)((long)_lon / 100L); // _lon is always positive here
    		 lon += (_lon - (lon * 100.0)) / 60.0;
    		 return d.equals("W")? -lon : lon;
    	 } else {
    		 return 180.0; // invalid longitude
    	 }
     }
     /***********************************************************
      * Fin de funciones adicionales para VT310
      ***********************************************************/
 

     /* based on the supplied packet data, return the remaining bytes to read in the packet */
     public int getActualPacketLength(byte packet[], int packetLen)
     {
    	 if (Constants.ASCII_PACKETS) {
    		 // (this actually won't be called if 'Constants.ASCII_PACKETS' is true).
    		 // ASCII packets - look for line terminator [see Constants.ASCII_LINE_TERMINATOR)]
    		 return ServerSocketThread.PACKET_LEN_LINE_TERMINATOR;  // read until line termination character
    		 //return ServerSocketThread.PACKET_LEN_END_OF_STREAM;  // read until end of stream, or maxlen
    	 } else {
    		 // BINARY packet - need to analyze 'packet[]' and determine actual packet length
    		 //return ServerSocketThread.PACKET_LEN_LINE_TERMINATOR; // <-- change this for binary packets
    		 //return ServerSocketThread.PACKET_LEN_END_OF_STREAM; // <-- Se elimino para permitir paquetes binarios
    		 if(4 > packet.length){
    			 return(0);
    		 }
    		 
    		 //Print.logInfo("Long: " + packetLen);
    		 //this.paint(null);
    		 
    		 
    		 //Payload p = new Payload(packet, 2,2);
    		 Payload p = new Payload(packet,2,2);
    		 int p_int	= p.readInt(2, 0);
    		 Print.logInfo("packet length: " + p_int);

    		 //int length = packetGetLength(packet);
    		 //System.out.println("JELO2");
    		 //return ServerSocketThread.PACKET_LEN_ASCII_LINE_TERMINATOR; // <-- change this for binary packets        	
    		 return(p_int);
    	 }

     }

     /* set session terminate after next packet handling */
     private void setTerminate()
     {
    	 this.terminate = true;
     }

     /* indicate that the session should terminate */
     // This method is called after each return from "getHandlePacket" to check to see
     // the current session should be closed.
     public boolean getTerminateSession()
     {
    	 return this.terminate;
     }

     // ------------------------------------------------------------------------
     // ------------------------------------------------------------------------

     /* return the initial packet sent to the Device after session is open */
     public byte[] getInitialPacket() 
    		 throws Exception
    		 {
    	 // At this point a connection from the client to the server has just been
    	 // initiated, and we have not yet received any data from the client.
    	 // If the client is expecting to receive an initial packet from the server at
    	 // the time that the client connects, then this is where the server can return
    	 // a byte array that will be transmitted to the client Device.
    	 return null;
    	 // Note: any returned response for "getInitialPacket()" is ignored for simplex/udp connections.
    	 // Returned UDP packets may be sent from "getHandlePacket" or "getFinalPacket".
    		 }

     /* workhorse of the packet handler */
     public byte[] getHandlePacket(byte pktBytes[]) 
     {
    	 if ((pktBytes != null) && (pktBytes.length > 0)) {
             
    		 System.out.println("");
         	
             if(!packetCheckCRC(pktBytes)){
             	return(null);
             }
             
             String device_id = packetGetDeviceIdStr(pktBytes);
             String s = StringTools.toStringValue(pktBytes).trim();
             
             //int pktLength = packetGetLength(pktBytes);
             //Print.logInfo("pktBytes.length: " + pktBytes.length +" pktLength:" + pktLength); 
             
             byte[] pktData_ba = new byte[pktBytes.length - 2-2-7-2-2-2];
             System.arraycopy(pktBytes, 2+2+7+2, pktData_ba, 0, pktBytes.length - 2-2-7-2-2-2);
             
             s = StringTools.toStringValue(pktData_ba).trim(); // remove leading/trailing spaces
             //Print.logInfo("Recv[HEX]: " + StringTools.toHexString(pktData_ba));
             //Print.logInfo("Recv[TXT]: " + s); // debug message

             byte rtn[] = null;
             
             int comandovt = packetGetDeviceCommand(pktBytes);
             //check if its a login request
             if(0x5000 == comandovt){
             	//don't process this packet, just send login confirmation
             	System.out.println("Equipo detectado: (" + device_id +"), enviando saludo...");
             	rtn = packetDeviceLoginAccept(pktBytes);
             	return(rtn);
             } else if (comandovt == 0x9955){
            	 System.out.println("[Reporte] " + tipoData +" : " + s);
            	 rtn = this.insertVT310(device_id, s, false, "");
             } else if (comandovt == 0x9999){
            	 String reportType="";
            	 reportType = StringTools.toHexString(pktBytes[13]);
            	 System.out.println("[Alarma] " + tipoData +" : " + s + " [" + reportType+"]");
            	 rtn = this.insertVT310(device_id, s, true, reportType);
             } else {
             	//don't know how to process this packet
            	 rtn = null;
            	 System.out.println("Solo se aceptan paquetes de posicion, descartando " + s);
            	 return(null);
             }

             

             return rtn;

         } else {

             /* no packet date received */
             Print.logInfo("Empty packet received ...");
             return null; // no return packets are expected

         }
     }

     /* final packet sent to Device before session is closed */
     public byte[] getFinalPacket(boolean hasError) 
    		 throws Exception
    		 {
    	 		//String trama = this.gpsDevice.getDescription();
    	 		return null;
    	 		//return trama.getBytes();
    		 }
     
     private byte[] insertVT310(String deviceID, String s, boolean alarma, String reportType)
     {
     	/**
     	 * 092432.000,A,0614.2404,S,10658.3312,E,0.00,,071010,,*01|0.8|42|0000
     	 * Reporte: 
     	 * 221530.193,A,1159.8200,S,07703.4023,W,000.0,248.3,300613,,,A*67|1.1|90.5|0000|0000,0000|017263498
     	 * Alarma:
     	 * R221605.092,A,1521.1529,S,07453.5510,W,052.6,343.7,300613,,,A*61|0.8|426|0010|0000,0000|02CC0006160BB718|1
     	 * slpit by '|'
     	 * 0:gprmc
     	 * 1:HDOP, in ASCII code, 0.5-99.9. HDOP is blank when the tracker has no GPS fix.
     	 * 2:Altitude, in algorism.
     	 * 3:State: Status of inputs and outputs: bitfield
     	 * 4:AD1,AD2: 10 bit analog input (only for voltage) for VT310 only, 0x0000~0x03ff in HEX, separated by , (comma).
     	 * 
     	 * GPRMC split by ','
     	 * 0:092432.000 - hhmmss.dd UTC time  hh = hours; mm = minutes; ss = seconds;  dd = decimal part of seconds
     	 * 1:A - GPS status indicator, A = valid, V = invalid
     	 * 2:0614.2404 - Latitude  xxmm.dddd  xx = degrees;  mm = minutes; dddd = decimal part of minutes
     	 * 3:S - hemishere [N|S]
     	 * 4:10658.3312 - Longitude yyymm.dddd yyy = degrees; mm = minutes; dddd = decimal part of minutes
     	 * 5:E - hemishere [E|W]
     	 * 6:0.00 - s.s Speed, in unit of knot. (1 knot = 1.852 km)
     	 * 7:     - h.h Heading, in unit of degree
     	 * 8:071010 - Date ddmmyy dd = date; mm = month; yy = year
     	 * 9:-d.d Magnetic variation Normally blank(the field can be ommited)
     	 * 10:-D Either character W or character E Normally blank(the field can be ommited)
     	 * 11:* - checksum delimiter:
     	 * 12:01- checksum
     	 * 
     	 * **/
    	 
    	 String   modemID    = deviceID;
         
         /* pre-validate */
         if (s == null) {
             Print.logError("String is null");
             return null;
         }

         /* parse to fields */
         //String fld[] = StringTools.parseString(s, ',');
         String fld[] = StringTools.parseStringArray(s, "|");
         /*String fld[] = StringTools.parseStringArray(s, ',');*/
         if ((fld == null) || (fld.length < 2)) {
             Print.logWarn("Invalid number of fields");
             return null;
         }
         
         // Deprecated String gprmc_fld[] = StringTools.parseString(fld[0], String.valueOf(","));
         String gprmc_fld[] = StringTools.parseStringArray(fld[0],",");
         if ((gprmc_fld == null) || (gprmc_fld.length < 9)) {
             Print.logWarn("gprmc_fld:Invalid number of fields");
             return null;
         }

         /* parse individual fields */
         long		timeL; //R221605.092
         if (alarma){
        	 String alarm = gprmc_fld[0].substring(1, gprmc_fld[0].length()-1);
        	 timeL 		= StringTools.parseLong(StringTools.parseStringArray(alarm, ",")[0], 0);
         } else {
        	 timeL 		= StringTools.parseLong(StringTools.parseStringArray(gprmc_fld[0], ",")[0], 0);
         }
         boolean 	validGPS   	= gprmc_fld[1].equalsIgnoreCase("A");
         double   	latitude   	= _parseLatitude(gprmc_fld[2],gprmc_fld[3]);
         double   	longitude  	= _parseLongitude(gprmc_fld[4],gprmc_fld[5]);
         double   	speedKPH   	= StringTools.parseDouble(gprmc_fld[6],0.0) * 1.852;//(1 knot = 1.852 km)
         double   	heading    	= StringTools.parseDouble(gprmc_fld[7],0.0);
         long		dateL 		= StringTools.parseLong(gprmc_fld[8], 0);
         double   	altitudeM  	= StringTools.parseDouble(fld[2], 0.0);  //meitrack says Altitude, in algorism. Go figure. ignored for now
         
         long     	fixtime 	= _getUTCSeconds(dateL, timeL);
         int      	statusCode 	= StatusCodes.STATUS_LOCATION;
         /* GPIO input */
         long     gpioInput  = -1L;
         
         if (!validGPS){
         	System.out.println("DATA GPS NO VALIDA!. --- Se descarta trama para evitar errores ---");
         	return null;
         }
                  
         /* no modemID? */
         if (StringTools.isBlank(modemID)) {
             Print.logWarn("ModemID not specified!");
             return null;
         }
         
         /* GPS Event */
    	 this.gpsEvent = this.createGPSEvent(modemID);
    	 if (this.gpsEvent == null) {
    		 // errors already displayed
    		 return null;
    	 }

    	 Device device = this.gpsEvent.getDevice();
    	 if (device == null) {
    		 return null;
    	 }
    	 String accountID 	= device.getAccountID();
    	 String devID		= device.getDeviceID();
    	 String uniqueID  	= device.getUniqueID();
    	 System.out.println("Registro ---> [" + accountID + "/" + devID + "/" + uniqueID + "]");
    	 System.out.println("Evento: [" + new DateTime(fixtime) + "]");
    	 
    	 /** Ajuste de Velocidad - Motion Start **/
    	 double maxSpeed	= device.getSpeedLimitKPH();
    	 if ((speedKPH > MINIMUM_SPEED_KPH) && (speedKPH < maxSpeed)) {
    		 if(device.getLastValidSpeed() <= MINIMUM_SPEED_KPH){
    			 statusCode = StatusCodes.STATUS_MOTION_START;
    		 }else{
    			 statusCode = StatusCodes.STATUS_MOTION_IN_MOTION;	 
    		 }
    		 
    	 } else if (speedKPH >= maxSpeed) {
    		 statusCode = StatusCodes.STATUS_MOTION_EXCESS_SPEED;
    	 } else if (speedKPH <= MINIMUM_SPEED_KPH) {
    		 statusCode = StatusCodes.STATUS_MOTION_STOP;
    	 }

    	 /* Alerta de Velocidad MTC - SUTRAN */
    	 speedKPH = this.gpsEvent.ajustaVelocidad(device.getDescription(), speedKPH);
    	 this.lastModemID 	=	modemID;
    	 
    	 
    	 if (alarma){
    		 System.out.println("Reporte: " + reportType);
    		 if (reportType.equalsIgnoreCase("01")){
    			 statusCode = StatusCodes.STATUS_PANIC_ON;
    		 } else if (reportType.equalsIgnoreCase("02")){
    			 statusCode = StatusCodes.STATUS_INPUT_ON_02;
    		 } else if (reportType.equalsIgnoreCase("03")){
    			 statusCode = StatusCodes.STATUS_INPUT_ON_03;
    		 } else if (reportType.equalsIgnoreCase("04")){
    			 gpioInput = 1;		
    			 statusCode = StatusCodes.STATUS_IGNITION_ON;		     		
    			 device.setLastInputState(gpioInput & 0xFFFFL); 
    		 } else if (reportType.equalsIgnoreCase("05")){
    			 gpioInput = 1;		
    			 statusCode = StatusCodes.STATUS_IGNITION_ON;		     		
    			 device.setLastInputState(gpioInput & 0xFFFFL); 
    		 } else if (reportType.equalsIgnoreCase("10")){
    			 statusCode = StatusCodes.STATUS_LOW_BATTERY;;
    		 } else if (reportType.equalsIgnoreCase("11")){
    			 statusCode = StatusCodes.STATUS_MOTION_EXCESS_SPEED;		     		
    		 } else if (reportType.equalsIgnoreCase("31")){
    			 statusCode = StatusCodes.STATUS_PANIC_OFF;		     		
    		 } else if (reportType.equalsIgnoreCase("32")){
    			 statusCode = StatusCodes.STATUS_INPUT_OFF_02;		     		
    		 } else if (reportType.equalsIgnoreCase("33")){
    			 statusCode = StatusCodes.STATUS_INPUT_OFF_03;		     		
    		 } else if (reportType.equalsIgnoreCase("34")){
    			 gpioInput = 0;	
    			 statusCode = StatusCodes.STATUS_IGNITION_OFF;		
    			 device.setLastInputState(gpioInput & 0xFFFFL);	     		
    		 } else if (reportType.equalsIgnoreCase("35")){
    			 gpioInput = 0;	
    			 statusCode = StatusCodes.STATUS_IGNITION_OFF;		
    			 device.setLastInputState(gpioInput & 0xFFFFL);	     		
    		 } else if (reportType.equalsIgnoreCase("50")){
    			 statusCode = StatusCodes.STATUS_POWER_FAILURE;
    		 } else if (reportType.equalsIgnoreCase("15")){
    			 statusCode = StatusCodes.STATUS_GEOBOUNDS_ENTER;
    		 } else if (reportType.equalsIgnoreCase("16")){
    			 statusCode = StatusCodes.STATUS_GEOBOUNDS_EXIT;
    		 } else if (reportType.equalsIgnoreCase("52")){
    			 statusCode = StatusCodes.STATUS_MOTION_HEADING;
    		 } else if (reportType.equalsIgnoreCase("52")){
    			 statusCode = StatusCodes.STATUS_MOTION_HEADING;
    		 }
    		 
    	 }
    	 
         this.gpsEvent.setTimestamp(fixtime);
         this.gpsEvent.setStatusCode(statusCode);
         this.gpsEvent.setLatitude(latitude);
         this.gpsEvent.setLongitude(longitude);
         this.gpsEvent.setSpeedKPH(speedKPH);
         this.gpsEvent.setHeading(heading);
         this.gpsEvent.setAltitude(altitudeM);
         this.gpsEvent.setSatelliteCount(8);
         this.gpsEvent.setBatteryLevel(1);
         //Print.logInfo(gpsEvent.toString());
         
         /* insert/return */
         if (this.parseInsertRecord_Common(this.gpsEvent)) {
             // change this to return any required acknowledgement (ACK) packets back to the device
        	
             return null;
         } else {
             return null;
         }
         
     }
     
     /* parse and insert data record (common) */
     private boolean parseInsertRecord_Common(GPSEvent gpsEv)
     {
    	 long   fixtime    = gpsEv.getTimestamp();
    	 int    statusCode = gpsEv.getStatusCode();
    	 Device dev        = gpsEv.getDevice(); // guaranteed non-null here

    	 /* invalid date? */
    	 if (fixtime <= 0L) {
    		 Print.logWarn("Invalid date/time");
    		 fixtime = DateTime.getCurrentTimeSec(); // default to now
    		 gpsEv.setTimestamp(fixtime);
    	 }

    	 /* valid lat/lon? */
    	 if (!gpsEv.isValidGeoPoint()) {
    		 Print.logWarn("Invalid lat/lon: " + gpsEv.getLatitude() + "/" + gpsEv.getLongitude());
    		 gpsEv.setLatitude(0.0);
    		 gpsEv.setLongitude(0.0);
    	 }
    	 GeoPoint geoPoint = gpsEv.getGeoPoint();

    	 /* minimum speed */
    	 if (gpsEv.getSpeedKPH() < MINIMUM_SPEED_KPH) {
    		 gpsEv.setSpeedKPH(0.0);
    		 gpsEv.setHeading(0.0);
    	 }

    	 /* estimate GPS-based odometer */
    	 double odomKM = 0.0; // set to available odometer from event record
    	 //if (this.gpsDevice.getLastEventTimestamp() < fixtime) {
    	 if (odomKM <= 0.0) {
    		 odomKM = (ESTIMATE_ODOMETER && geoPoint.isValid())? 
    				 this.gpsDevice.getNextOdometerKM(geoPoint) : 
    					 this.gpsDevice.getLastOdometerKM();
    	 } else {
    		 odomKM = dev.adjustOdometerKM(odomKM);
    	 }
    	 //}
    	 Print.logInfo("Odometer KM: " + odomKM);
    	 gpsEv.setOdometerKM(odomKM);

    	 /* simulate Geozone arrival/departure */
    	 if (SIMEVENT_GEOZONES && geoPoint.isValid()) {
    		 java.util.List<Device.GeozoneTransition> zone = dev.checkGeozoneTransitions(fixtime, geoPoint);
    		 if (zone != null) {
    			 for (Device.GeozoneTransition z : zone) {
    				 gpsEv.insertEventData(z.getTimestamp(), z.getStatusCode(), z.getGeozone());
    				 Print.logInfo("Geozone    : " + z);
    			 }
    		 }
    	 }

    	 /* digital input change events */
    	 if (gpsEv.hasInputMask() && (gpsEv.getInputMask() >= 0L)) {
    		 long gpioInput = gpsEv.getInputMask();
    		 if (SIMEVENT_DIGITAL_INPUTS > 0L) {
    			 // The current input state is compared to the last value stored in the Device record.
    			 // Changes in the input state will generate a synthesized event.
    			 long chgMask = (dev.getLastInputState() ^ gpioInput) & SIMEVENT_DIGITAL_INPUTS;
    			 if (chgMask != 0L) {
    				 // an input state has changed
    				 for (int b = 0; b <= 15; b++) {
    					 long m = 1L << b;
    					 if ((chgMask & m) != 0L) {
    						 // this bit changed
    						 int  inpCode = ((gpioInput & m) != 0L)? InputStatusCodes_ON[b] : InputStatusCodes_OFF[b];
    						 long inpTime = fixtime;
    						 gpsEv.insertEventData(inpTime, inpCode);
    						 Print.logInfo("GPIO : " + StatusCodes.GetDescription(inpCode,null));
    					 }
    				 }
    			 }
    		 }
    		 dev.setLastInputState(gpioInput & 0xFFFFL); // FLD_lastInputState
    	 }

    	 /* create/insert standard event */
    	 gpsEv.insertEventData(fixtime, statusCode);

    	 /* save Device changes */
    	 gpsEv.updateDevice();

    	 /* return success */
    	 return true;

     }

     /**
      *** Initialize runtime configuration
      **/
     public static void configInit() 
     {
    	 DCServerConfig dcsc     = Main.getServerConfig(null);
    	 if (dcsc == null) {
    		 Print.logWarn("DCServer not found: " + Main.getServerName());
    		 return;
    	 }

    	 /* custom */
    	 DATA_FORMAT_OPTION      = dcsc.getIntProperty(Main.ARG_FORMAT, DATA_FORMAT_OPTION);

    	 /* common */
    	 MINIMUM_SPEED_KPH       = dcsc.getMinimumSpeedKPH(MINIMUM_SPEED_KPH);
    	 ESTIMATE_ODOMETER       = dcsc.getEstimateOdometer(ESTIMATE_ODOMETER);
    	 SIMEVENT_GEOZONES       = dcsc.getSimulateGeozones(SIMEVENT_GEOZONES);
    	 SIMEVENT_DIGITAL_INPUTS = dcsc.getSimulateDigitalInputs(SIMEVENT_DIGITAL_INPUTS) & 0xFFFFL;

     }

     // ------------------------------------------------------------------------
     // ------------------------------------------------------------------------
     // ------------------------------------------------------------------------
     // Once you have modified this example 'template' server to parse your particular
     // Device packets, you can also use this source module to load GPS data packets
     // which have been saved in a file.  To run this module to load your save GPS data
     // packets, start this command as follows:
     //   java -cp <classpath> org.opengts.servers.template.TrackClientPacketHandler {options}
     // Where your options are one or more of 
     //   -insert=[true|false]    Insert parse records into EventData
     //   -format=[1|2]           Data format
     //   -debug                  Parse internal sample data
     //   -parseFile=<file>       Parse data from specified file

     private static int _usage()
     {
    	 String cn = StringTools.className(TrackClientPacketHandler.class);
    	 Print.sysPrintln("Test/Load Device Communication Server");
    	 Print.sysPrintln("Usage:");
    	 Print.sysPrintln("  $JAVA_HOME/bin/java -classpath <classpath> %s {options}", cn);
    	 Print.sysPrintln("Options:");
    	 Print.sysPrintln("  -insert=[true|false]    Insert parsed records into EventData");
    	 Print.sysPrintln("  -format=[1|2]           Data format");
    	 Print.sysPrintln("  -debug                  Parse internal sample/debug data (if any)");
    	 Print.sysPrintln("  -parseFile=<file>       Parse data from specified file");
    	 return 1;
     }

     /**
      *** Main entry point used for debug purposes only
      **/
     public static int _main(boolean fromMain)
     {

    	 /* default options */
    	 INSERT_EVENT = RTConfig.getBoolean(Main.ARG_INSERT, DFT_INSERT_EVENT);
    	 if (!INSERT_EVENT) {
    		 Print.sysPrintln("Warning: Data will NOT be inserted into the database");
    	 }

    	 /* create client packet handler */
    	 TrackClientPacketHandler tcph = new TrackClientPacketHandler();

    	 /* DEBUG sample data */
    	 if (RTConfig.getBoolean(Main.ARG_DEBUG,false)) {
    		 String data[] = null;
    		 switch (DATA_FORMAT_OPTION) {
    		 case  1: data = new String[] {
    				 "123456789012345,2006/09/05,07:47:26,35.3640,-142.2958,27.0,224.8",
    		 }; break;
    		 case  2: data = new String[] {
    				 "account/device/$GPRMC,025423.494,A,3709.0642,N,14207.8315,W,12.09,108.52,200505,,*2E",
    				 "/device/$GPRMC,025423.494,A,3709.0642,N,14207.8315,W,12.09,108.52,200505,,*2E",
    		 }; break;
    		 case  3: data = new String[] {
    				 "2,123,1234567890,0,20101223,110819,1,2.1,39.1234,-142.1234,33,227,1800",
    		 }; break;
    		 case  9: data = new String[] {
    				 "mid=123456789012345 lat=39.12345 lon=-142.12345 kph=123.0"
    		 }; break;
    		 default:
    			 Print.sysPrintln("Unrecognized Data Format: %d", DATA_FORMAT_OPTION);
    			 return _usage();
    		 }
    		 for (int i = 0; i < data.length; i++) {
    			 tcph.getHandlePacket(data[i].getBytes());
    		 }
    		 return 0;
    	 }

    	 /* 'parseFile' specified? */
    	 if (RTConfig.hasProperty(Main.ARG_PARSEFILE)) {

    		 /* get input file */
    		 File parseFile = RTConfig.getFile(Main.ARG_PARSEFILE,null);
    		 if ((parseFile == null) || !parseFile.isFile()) {
    			 Print.sysPrintln("Data source file not specified, or does not exist.");
    			 return _usage();
    		 }

    		 /* open file */
    		 FileInputStream fis = null;
    		 try {
    			 fis = new FileInputStream(parseFile);
    		 } catch (IOException ioe) {
    			 Print.logException("Error openning input file: " + parseFile, ioe);
    			 return 2;
    		 }

    		 /* loop through file */
    		 try {
    			 // records are assumed to be terminated by CR/NL 
    			 for (;;) {
    				 String data = FileTools.readLine(fis);
    				 if (!StringTools.isBlank(data)) {
    					 tcph.getHandlePacket(data.getBytes());
    				 }
    			 }
    		 } catch (EOFException eof) {
    			 Print.sysPrintln("");
    			 Print.sysPrintln("***** End-Of-File *****");
    		 } catch (IOException ioe) {
    			 Print.logException("Error reaading input file: " + parseFile, ioe);
    		 } finally {
    			 try { fis.close(); } catch (Throwable th) {/* ignore */}
    		 }

    		 /* done */
    		 return 0;

    	 }

    	 /* no options? */
    	 return _usage();

     }

     /**
      *** Main entry point used for debug purposes only
      **/
     public static void main(String argv[])
     {
    	 DBConfig.cmdLineInit(argv,false);
    	 TrackClientPacketHandler.configInit();
    	 System.exit(TrackClientPacketHandler._main(false));
     }

}
