package org.opengts.servers.gt06;

import java.util.*;
import java.io.*;
import java.net.*;
import org.opengts.util.*;
import org.opengts.db.*;
import org.opengts.db.tables.*;
import org.opengts.dbtools.DBException;
import org.opengts.dbtypes.DTIPAddrList;

import org.opengts.servers.*;

/**
*** <code>TrackClientPacketHandler</code> - This module contains the general
*** "business logic" for parsing incoming data packets from the remote tracking
*** Device.
**/

public class TrackClientPacketHandler
    extends AbstractClientPacketHandler
{
    public  static 		int	DATA_FORMAT_OPTION				= 1;

    //    vehicle value.
    public  static		boolean ESTIMATE_ODOMETER          = true;
    
    /* simulate geozone arrival/departure */
    // (enable to insert simulated Geozone arrival/departure EventData records)
    public  static       boolean SIMEVENT_GEOZONES          = true;
    
    /* simulate digital input changes */
    public  static       long    SIMEVENT_DIGITAL_INPUTS    = 0x0000L; // 0xFFFFL;

    /* flag indicating whether data should be inserted into the DB */
    // should be set to 'true' for production.
    private static       boolean DFT_INSERT_EVENT           = true;
    private static       boolean INSERT_EVENT               = DFT_INSERT_EVENT;

    /* update Device record */
    // (enable to update Device record with current IP address and last connect time)
  //private static       boolean UPDATE_DEVICE              = false;
    
    /* minimum acceptable speed value */
    // Speeds below this value should be considered 'stopped'
    public  static       double  MINIMUM_SPEED_KPH          = 0.0;

    /* Knot/Kilometer conversions */
    public static final double  KILOMETERS_PER_KNOT     = 1.85200000;
    public static final double  KNOTS_PER_KILOMETER     = 1.0 / KILOMETERS_PER_KNOT;

    // ------------------------------------------------------------------------

    /* Ingore $GPRMC checksum? */
    // (only applicable for data formats that include NMEA-0183 formatted event records)
    //private static       boolean IGNORE_NMEA_CHECKSUM       = false;

    // ------------------------------------------------------------------------

    /* GMT/UTC timezone */
    private static final TimeZone gmtTimezone               = DateTime.getGMTTimeZone();

    // ------------------------------------------------------------------------
    
    private 			String 		tipoData				=	"";

    /* GTS status codes for Input-On events */
    private static final int InputStatusCodes_ON[] = new int[] {
        StatusCodes.STATUS_INPUT_ON_00,
        StatusCodes.STATUS_INPUT_ON_01,
        StatusCodes.STATUS_INPUT_ON_02,
        StatusCodes.STATUS_INPUT_ON_03,
        StatusCodes.STATUS_INPUT_ON_04,
        StatusCodes.STATUS_INPUT_ON_05,
        StatusCodes.STATUS_INPUT_ON_06,
        StatusCodes.STATUS_INPUT_ON_07,
        StatusCodes.STATUS_INPUT_ON_08,
        StatusCodes.STATUS_INPUT_ON_09,
        StatusCodes.STATUS_INPUT_ON_10,
        StatusCodes.STATUS_INPUT_ON_11,
        StatusCodes.STATUS_INPUT_ON_12,
        StatusCodes.STATUS_INPUT_ON_13,
        StatusCodes.STATUS_INPUT_ON_14,
        StatusCodes.STATUS_INPUT_ON_15
    };

    /* GTS status codes for Input-Off events */
    private static final int InputStatusCodes_OFF[] = new int[] {
        StatusCodes.STATUS_INPUT_OFF_00,
        StatusCodes.STATUS_INPUT_OFF_01,
        StatusCodes.STATUS_INPUT_OFF_02,
        StatusCodes.STATUS_INPUT_OFF_03,
        StatusCodes.STATUS_INPUT_OFF_04,
        StatusCodes.STATUS_INPUT_OFF_05,
        StatusCodes.STATUS_INPUT_OFF_06,
        StatusCodes.STATUS_INPUT_OFF_07,
        StatusCodes.STATUS_INPUT_OFF_08,
        StatusCodes.STATUS_INPUT_OFF_09,
        StatusCodes.STATUS_INPUT_OFF_10,
        StatusCodes.STATUS_INPUT_OFF_11,
        StatusCodes.STATUS_INPUT_OFF_12,
        StatusCodes.STATUS_INPUT_OFF_13,
        StatusCodes.STATUS_INPUT_OFF_14,
        StatusCodes.STATUS_INPUT_OFF_15
    };

    // ------------------------------------------------------------------------
    // ------------------------------------------------------------------------

    /* TCP session ID */
    private String          sessionID                   = null;

    /* common GPSEvent instance */
    private GPSEvent        gpsEvent                    = null;

    /* Device record */
    private Device          gpsDevice                   = null;
    private String          lastModemID                 = null;
    private	String			gt06DeviceID				= null;

    /* Session 'terminate' indicator */
    // This value should be set to 'true' when this server has determined that the
    // session should be terminated.  For instance, if this server finishes communication
    // with the Device or if parser finds a fatal error in the incoming data stream 
    // (ie. invalid Account/Device, or unrecognizable data).
    private boolean         terminate                   = false;

    /* session IP address */
    // These values will be set for you by the incoming session to indicate the 
    // originating IP address.
    private String          ipAddress                   = null;
    private int             clientPort                  = 0;

    /* packet handler constructor */
    public TrackClientPacketHandler() 
    {
        super();
        //Print.logStackTrace("new TrackClientPacketHandler ...");
    }

    // ------------------------------------------------------------------------

    /* callback when session is starting */
    // this method is called at the beginning of a communication session
    public void sessionStarted(InetAddress inetAddr, boolean isTCP, boolean isText)
    {
        super.sessionStarted(inetAddr, isTCP, isText);
        super.clearTerminateSession();

        /* init */
        this.ipAddress        = (inetAddr != null)? inetAddr.getHostAddress() : null;
        this.clientPort       = this.getSessionInfo().getRemotePort();
        if (this.isDuplex()) {
        	this.tipoData="TCP";
            //Print.logInfo("Inicio TCP: " + this.ipAddress + " [" + new DateTime() + "]");
        	//	Print.logInfo("========================================================");
        } else {
        	this.tipoData="UDP";
        	//Print.logInfo("Inicio UDP: " + this.ipAddress + " [" + new DateTime() + "]");
        	//Print.logInfo("========================================================");
        }

    }

    /* callback when session is terminating */
    // this method is called at the end of a communication session
    public void sessionTerminated(Throwable err, long readCount, long writeCount)
    {
        super.sessionTerminated(err, readCount, writeCount);
    }

    /* callback to return the TCP session id */
    public String getSessionID()
    {
        if (!StringTools.isBlank(this.sessionID)) {
            return this.sessionID;
        } else
        if (this.gpsDevice != null) {
            return CreateTcpSessionID(this.gpsDevice);
        } else
        if (this.gpsEvent != null) {
            return CreateTcpSessionID(this.gpsEvent.getDevice());
        } else {
            return null;
        }
    }

    // ------------------------------------------------------------------------

    public static final boolean USE_STANDARD_TCP_SESSION_ID = true;

    /* get TCP session ID */
    public static String GetTcpSessionID(Device dev)
    {
        if (USE_STANDARD_TCP_SESSION_ID || (dev == null)) {
            return DCServerFactory.getTcpSessionID(dev);
        } else {
            // Custom example for extracting the session ID from the Device 
            // UniqueID (instead of the DeviceID)
            String uid = dev.getUniqueID();
            int    u   = uid.lastIndexOf("_");
            if (u < 0) {
                // no "_" character, fallback to standard TCP sessionID
                return DCServerFactory.getTcpSessionID(dev);
            } else {
                String sessID = dev.getAccountID() + "/" + uid.substring(0,u+1);
                return sessID;
            }
        }
    }

    /* create TCP session ID */
    public static String CreateTcpSessionID(Device dev)
    {
        if (USE_STANDARD_TCP_SESSION_ID || (dev == null)) {
            return DCServerFactory.createTcpSessionID(dev);
        } else {
            String uid = dev.getUniqueID();
            int    u   = uid.lastIndexOf("_");
            if (u < 0) {
                // no "_" character, fallback to standard TCP sessionID
                return DCServerFactory.createTcpSessionID(dev);
            } else {
                String sessID = dev.getAccountID() + "/" + uid.substring(0,u+1);
                dev.setLastTcpSessionID(sessID);
                return sessID;
            }
        }
    }

     /**
     *** Create GPSEvent from the ModemID
     *** @return The created GPSEvent instance, or null if the ModemID is invalid
     **/
     private GPSEvent createGPSEvent(String modemID)
     {
         DCServerConfig dcserver = Main.getServerConfig(null);

         /* create/load device */
         if (this.gpsDevice != null) {
             if (!modemID.equals("*") && 
                 ((this.lastModemID == null) || !this.lastModemID.equals(modemID))) { // fix [B10]
                 Print.logError("New MobileID does not match previously loaded Device");
                 //return this.gpsEvent;
                 return null;
             }
             this.gpsEvent  = new GPSEvent(dcserver, this.ipAddress, this.clientPort, this.gpsDevice);
         } else {
             if (StringTools.isBlank(modemID) || modemID.equals("*")) {
                 Print.logWarn("ModemID not specified!");
                 return null;
             }
             this.gpsEvent  = new GPSEvent(dcserver, this.ipAddress, this.clientPort, modemID);
             this.gpsDevice = this.gpsEvent.getDevice(); // may still be null
         }

         /* no Device? */
         if (this.gpsDevice == null) {
             // errors already displayed
             return null;
         }

         /* create session ID */
         this.sessionID = CreateTcpSessionID(this.gpsDevice);

         /* return GPSEvent */
         return this.gpsEvent; // non-null

     }

    /* based on the supplied packet data, return the remaining bytes to read in the packet */
    public int getActualPacketLength(byte packet[], int packetLen)
    {
        if (Constants.ASCII_PACKETS) {
            // (this actually won't be called if 'Constants.ASCII_PACKETS' is true).
            // ASCII packets - look for line terminator [see Constants.ASCII_LINE_TERMINATOR)]
            return ServerSocketThread.PACKET_LEN_LINE_TERMINATOR;  // read until line termination character
            //return ServerSocketThread.PACKET_LEN_END_OF_STREAM;  // read until end of stream, or maxlen
        } else {
            // BINARY packet - need to analyze 'packet[]' and determine actual packet length
            //return ServerSocketThread.PACKET_LEN_LINE_TERMINATOR; // <-- change this for binary packets
        	return ServerSocketThread.PACKET_LEN_END_OF_STREAM; // <-- change this for binary packets
        }
        
    }

    /* set session terminate after next packet handling */
    private void setTerminate()
    {
        this.terminate = true;
    }
    
    /* indicate that the session should terminate */
    // This method is called after each return from "getHandlePacket" to check to see
    // the current session should be closed.
    public boolean getTerminateSession()
    {
        return this.terminate;
    }

    // ------------------------------------------------------------------------
    // ------------------------------------------------------------------------

    /* return the initial packet sent to the Device after session is open */
    public byte[] getInitialPacket() 
        throws Exception
    {
        // At this point a connection from the client to the server has just been
        // initiated, and we have not yet received any data from the client.
        // If the client is expecting to receive an initial packet from the server at
        // the time that the client connects, then this is where the server can return
        // a byte array that will be transmitted to the client Device.
        return null;
        // Note: any returned response for "getInitialPacket()" is ignored for simplex/udp connections.
        // Returned UDP packets may be sent from "getHandlePacket" or "getFinalPacket".
    }

    /* workhorse of the packet handler */
    public byte[] getHandlePacket(byte pktBytes[]){

    	String p = StringTools.toHexString(pktBytes).trim();
    	System.out.println("");
    	System.out.println("H: "+ p.toString());

        if ((pktBytes != null) && (pktBytes.length > 0)) {
            
            Print.logInfo("Recv[HEX]: " + StringTools.toHexString(pktBytes));

            /* parse/insert event */
            byte rtn[] = null;
            CRC16 crcc 	= 	new CRC16();
        	byte[] t	=	new byte[4];
        	byte[] f	=	new byte[10];
        	
        	Print.logInfo("Head: " + StringTools.toHexString(pktBytes[0])+StringTools.toHexString(pktBytes[1]));
        	Print.logInfo("Length: " + StringTools.toHexString(pktBytes[2]));//25h 
        	Print.logInfo("Protocol: " + StringTools.toHexString(pktBytes[3]));
        	Print.logInfo("serial: " + StringTools.toHexString(pktBytes[pktBytes[2]-6+5])+StringTools.toHexString(pktBytes[pktBytes[2]-5+5]));
        	Print.logInfo("crc: " + StringTools.toHexString(pktBytes[pktBytes[2]-4+5])+StringTools.toHexString(pktBytes[pktBytes[2]-3+5]));
        	Print.logInfo("stop: " + StringTools.toHexString(pktBytes[pktBytes[2]-2+5])+StringTools.toHexString(pktBytes[pktBytes[2]-1+5]));
        	
        	String protocol	=	StringTools.toHexString(pktBytes[3]);
        	System.out.println("Prot: " + protocol);
        	
//        	if(pktBytes[3]==01 || pktBytes[3]==13){
//        	}
        	
        	if	(pktBytes[3]==0x01 || pktBytes[3]==01 || pktBytes[3]==13){
        		this.gt06DeviceID=(StringTools.toHexString(pktBytes[4]) + StringTools.toHexString(pktBytes[5])+ 
        				StringTools.toHexString(pktBytes[6])+ StringTools.toHexString(pktBytes[7])+ 
        				StringTools.toHexString(pktBytes[8])+StringTools.toHexString(pktBytes[9])+
        				StringTools.toHexString(pktBytes[10])+StringTools.toHexString(pktBytes[11]));
        		System.out.println("IDTracker: " + this.gt06DeviceID.toLowerCase().trim());
        		Print.logInfo("");
        		t[0]=0x05;
        		t[1]=pktBytes[3];
        		t[2]=pktBytes[pktBytes[2]-6+5];
        		t[3]=pktBytes[pktBytes[2]-5+5];
        		crcc.set(t);

        		short x=crcc.get();

        		f[0]=0x78;
        		f[1]=0x78;
        		f[2]=0x05;
        		f[3]=pktBytes[3];
        		f[4]=pktBytes[pktBytes[2]-6+5];
        		f[5]=pktBytes[pktBytes[2]-5+5];
        		f[6]=(byte)((x >> 8) & 0xff);
        		f[7]=(byte)(x & 0xff);
        		f[8]=0x0d;
        		f[9]=0x0a;

        		//Print.logInfo("CRCC: " + StringTools.toHexString(crcc.get()));
        		//return f;
        		rtn = f;
        	} else if (pktBytes[3]==0x13){
        		Print.logInfo("13: ");
        		Print.logInfo("Device info: " + StringTools.toHexString(pktBytes[4]));
        		Print.logInfo("Voltage degree: " + StringTools.toHexString(pktBytes[5]));
        		Print.logInfo("GSM Signal: " + StringTools.toHexString(pktBytes[6]));
        		Print.logInfo("");
        	} else if (pktBytes[3]==0x16){
        		rtn = this.insertDataGT06(pktBytes, true); 
        		
        	} else if (pktBytes[3]==0x12){
        		rtn = this.insertDataGT06(pktBytes, false); 
                
        	}
        	
        	return rtn; // no return packets are expected

        } else {

            /* no packet date received */
            Print.logInfo("Empty packet received ...");
            return null; // no return packets are expected

        }
        
    	
    }

    /* final packet sent to Device before session is closed */
    public byte[] getFinalPacket(boolean hasError) 
        throws Exception
    {
        return null;
    }
    
    /* parse and insert data record */
    private byte[] insertDataGT06(byte packet[], boolean isAlert){
    	
    	if (!isAlert){
    		Print.logInfo("12: ");
    		Print.logInfo("Date: " + StringTools.toHexString(packet[4]) + StringTools.toHexString(packet[5]) + StringTools.toHexString(packet[6]) + StringTools.toHexString(packet[7]) + StringTools.toHexString(packet[8]) + StringTools.toHexString(packet[9]));
    		Print.logInfo("GPS info: " + StringTools.toHexString(packet[10]));
    		Print.logInfo("Latitude: " + StringTools.toHexString(packet[11]) + StringTools.toHexString(packet[12]) + StringTools.toHexString(packet[13]) + StringTools.toHexString(packet[14]));
    		Print.logInfo("Longitude: " + StringTools.toHexString(packet[15]) + StringTools.toHexString(packet[16]) + StringTools.toHexString(packet[17]) + StringTools.toHexString(packet[18]));
    		Print.logInfo("Speed: " + StringTools.toHexString(packet[19]));
    		Print.logInfo("Status/Course: " + StringTools.toHexString(packet[20]) + "/" + StringTools.toHexString(packet[21]));

    		// gps sats
    		int		satsCount 	= StringTools.parseInt(StringTools.toHexString(packet[10]), 0);
    		//lat
    		int 	lati 		= ((packet[11] & 0xFF) << 24) + ((packet[12] & 0xFF) << 16) + ((packet[13] & 0xFF) << 8) + (packet[14] & 0xFF);
    		double 	latd 		= (double)lati/30000;
    		double 	latd2 		= latd/60;
    		String 	latString 	= Double.toString(latd2);
    		double  latitude   	= StringTools.parseDouble(latString,0.0);
    		Print.logInfo("lat: " + latString);

    		//	long
    		int 	longi 		= ((packet[15] & 0xFF) << 24) + ((packet[16] & 0xFF) << 16) + ((packet[17] & 0xFF) << 8) + (packet[18] & 0xFF);
    		double 	longd 		= (double)longi/30000;
    		double 	longd2 		= longd/60;
    		String 	longString 	= Double.toString(longd2);
    		double	longitude  	= StringTools.parseDouble(longString,0.0);
    		Print.logInfo("long: " + longString);
    		
    		//	unixtime
    		int 	YY	=(packet[4]&0xFF);
    		int 	MM	=(packet[5]&0xFF);
    		int 	DD	=(packet[6]&0xFF);
    		int 	hh	=(packet[7]&0xFF);
    		int 	mm	=(packet[8]&0xFF);
    		int 	ss	=(packet[9]&0xFF);
    		
    		if (YY < 100) { YY += 2000; }
    		DateTime dt = new DateTime(gmtTimezone,YY,MM,DD,hh,mm,ss);
    		long     fixtime  = dt.getTimeSec();
    		Print.logInfo("time: " + fixtime);

    		//	speed
    		double   speedKPH   = StringTools.parseDouble(packet[19],0.0);
    		Print.logInfo("speed: " + speedKPH);
    		int     statusCode  = StatusCodes.STATUS_LOCATION;
    		if (speedKPH>5){
    			statusCode= StatusCodes.STATUS_MOTION_IN_MOTION;
    		} else{
    			statusCode= StatusCodes.STATUS_MOTION_STOP;
    		}

    		int a	= ((packet[20]&0xff)<<8);
    		a+=(packet[21]&0xff);
    		int a2	= (a&0xFC00)>>10; //status 6 bit
    		int a1	= a&0x03ff; //compas 10 bit 
    		double	heading  = (double)a1;
    		Print.logInfo("status: "+ a2 +" compass: " + heading);


    		boolean longEW=(a2 & (1 << 1)) != 0 ? true : false;
    		char WE=(longEW == true ? 'W' : 'E');
    		Print.logInfo(""+WE);
    		longitude=(longEW == true ? -longitude : longitude);

    		boolean latSN=(a2 & (1 << 0)) != 0 ? true : false;
    		char NS=(latSN == true ? 'N' : 'S');
    		Print.logInfo(""+NS);
    		latitude=(latSN == true ? latitude : -latitude);

    		Print.logInfo("ID Tracker: " + gt06DeviceID);

    		/* GPS Event */
            this.gpsEvent = this.createGPSEvent(gt06DeviceID);//modemID);
            if (this.gpsEvent == null) {
                // errors already displayed
                return null;
            }
        	
        	Device device = this.gpsEvent.getDevice();
        	if (device == null) {
        		return null;
        	}
        	String accountID = device.getAccountID();
            String deviceID  = device.getDeviceID();
            String uniqueID  = device.getUniqueID();
            System.out.println("Registro ---> [" + accountID + "/" + deviceID + "/" + uniqueID + "]");
            System.out.println("Evento: [" + new DateTime(fixtime) + "]");
        	
            /** Ajuste de Velocidad - Motion Start **/
            double maxSpeed	= device.getSpeedLimitKPH();
            if ((speedKPH > MINIMUM_SPEED_KPH) && (speedKPH < maxSpeed)) {
            	if(device.getLastValidSpeed() <= MINIMUM_SPEED_KPH){
            		statusCode = StatusCodes.STATUS_MOTION_START;
            	}else{
            		statusCode = StatusCodes.STATUS_MOTION_IN_MOTION;	 
            	}

            } else if (speedKPH >= maxSpeed) {
            	statusCode = StatusCodes.STATUS_MOTION_EXCESS_SPEED;
            } else if (speedKPH <= MINIMUM_SPEED_KPH) {
            	statusCode = StatusCodes.STATUS_MOTION_STOP;
            }

        	/* Alerta de Velocidad MTC - SUTRAN */
        	speedKPH = this.gpsEvent.ajustaVelocidad(device.getDescription(), speedKPH);
        	this.lastModemID 	=	gt06DeviceID;//modemID;

        	this.gpsEvent.setTimestamp(fixtime);
        	this.gpsEvent.setStatusCode(statusCode);
        	this.gpsEvent.setLatitude(latitude);
        	this.gpsEvent.setLongitude(longitude);
        	this.gpsEvent.setSpeedKPH(speedKPH);
        	this.gpsEvent.setHeading(heading);
        	this.gpsEvent.setSatelliteCount(satsCount);
        	this.gpsEvent.setBatteryLevel(100);
        	//this.gpsEvent.setAltitude(altitudeM);
        	//this.gpsEvent.setSignalStrength(rssi);
        	
        	/* insert/return */
        	if (this.parseInsertRecord_Common(this.gpsEvent)) {
        		// change this to return any required acknowledgement (ACK) packets back to the device
        		return null;
        	} else {
        		return null;
        	}
    	}else{
    		Print.logInfo("16: ");
    		Print.logInfo("Device info: " + StringTools.toHexString(packet[4]));
    		Print.logInfo("Fecha: " + StringTools.toHexString(packet[4]) + StringTools.toHexString(packet[5]) + StringTools.toHexString(packet[6]) + StringTools.toHexString(packet[7]) + StringTools.toHexString(packet[8]) + StringTools.toHexString(packet[9]));
    		Print.logInfo("GPS info: " + StringTools.toHexString(packet[10]));
    		Print.logInfo("Latitude: " + StringTools.toHexString(packet[11]) + StringTools.toHexString(packet[12]) + StringTools.toHexString(packet[13]) + StringTools.toHexString(packet[14]));
    		Print.logInfo("Longitude: " + StringTools.toHexString(packet[15]) + StringTools.toHexString(packet[16]) + StringTools.toHexString(packet[17]) + StringTools.toHexString(packet[18]));
    		Print.logInfo("Speed: " + StringTools.toHexString(packet[19]));
    		Print.logInfo("Status/Course: " + StringTools.toHexString(packet[20]) + "/" + StringTools.toHexString(packet[21]));
    		Print.logInfo("Info General: " + StringTools.toHexString(packet[31]));
    		Print.logInfo("Voltage degree: " + StringTools.toHexString(packet[32]));
    		Print.logInfo("GSM Signal: " + StringTools.toHexString(packet[33]));
    		Print.logInfo("Alarm Language: " + StringTools.toHexString(packet[34]) + StringTools.toHexString(packet[35]));
    		
    		// gps sats
    		int		satsCount 	= StringTools.parseInt(StringTools.toHexString(packet[10]), 0);
    		// General status
    		//int		IntGenStus		= StringTools.parseInt(StringTools.toHexString(packet[31]), 0);
    		String	GenStus			= StringTools.toBinaryString(packet[31]);
    		//System.out.println("-----  INFO:  " + GenStus); //00100010
    		boolean panic 		= false;
    		boolean powerCut 	= false;
    		boolean ignition	= false;
    		//System.out.println("BOTON: " +GenStus.substring(2,5));
    		if (GenStus.substring(2,5).equalsIgnoreCase("100")){
    			panic = true;
    		} else if (GenStus.substring(2,5).equalsIgnoreCase("010")){
    			powerCut = true;
    		}

    		if (GenStus.substring(6).equalsIgnoreCase("1")){
    			ignition=true;
    		} else if (GenStus.substring(6).equalsIgnoreCase("0")){
    			ignition=false;
    		}
    		
    		// Voltage
    		int 	voltLevel	= StringTools.parseInt(StringTools.toHexString(packet[32]), 0);
    		voltLevel = voltLevel * 16;
    		// RSSI
    		int 	signalRSSI	= StringTools.parseInt(StringTools.toHexString(packet[33]), 0);
    		
    		//lat
    		int lati = ((packet[11] & 0xFF) << 24) + ((packet[12] & 0xFF) << 16) + ((packet[13] & 0xFF) << 8) + (packet[14] & 0xFF);
    		double latd = (double)lati/30000;
    		double latd2 = latd/60;
    		String latString = Double.toString(latd2);
    		Print.logInfo("lat: " + latString);
    		double   latitude   = StringTools.parseDouble(latString,0.0);

    		//	long
    		int longi = ((packet[15] & 0xFF) << 24) + ((packet[16] & 0xFF) << 16) + ((packet[17] & 0xFF) << 8) + (packet[18] & 0xFF);
    		double longd = (double)longi/30000;
    		double longd2 = longd/60;
    		String longString = Double.toString(longd2);
    		Print.logInfo("long: " + longString);
    		double   longitude  = StringTools.parseDouble(longString,0.0);

    		//	unixtime
    		int YY=(packet[4]&0xFF);
    		int MM=(packet[5]&0xFF);
    		int DD=(packet[6]&0xFF);
    		int hh=(packet[7]&0xFF);
    		int mm=(packet[8]&0xFF);
    		int ss=(packet[9]&0xFF);
    		if (YY < 100) { YY += 2000; }
    		DateTime dt = new DateTime(gmtTimezone,YY,MM,DD,hh,mm,ss);
    		long     fixtime  = dt.getTimeSec();
    		Print.logInfo("time: " + fixtime);

    		//	speed
    		double   speedKPH   = StringTools.parseDouble(packet[19],0.0);
    		Print.logInfo("speed: " + speedKPH);

    		int     statusCode  = StatusCodes.STATUS_LOCATION;

    		int a=((packet[20]&0xff)<<8);
    		a+=(packet[21]&0xff);
    		int a2=(a&0xFC00)>>10; //status 6 bit
    		int a1=a&0x03ff; //compas 10 bit 
    		double   heading  = (double)a1;
    		Print.logInfo("status: "+ a2 +" compass: " + heading);

    		///Thanks vebraga
    		//status 6bit 1-sn,2-ew bit
    		//0-S 1-N
    		//0-E 1-W
    		boolean longEW=(a2 & (1 << 1)) != 0 ? true : false;
    		char WE=(longEW == true ? 'W' : 'E');
    		Print.logInfo(""+WE);
    		//double WE1=(longEW == true ? -longitude : longitude);
    		//Print.logInfo(""+WE1);
    		longitude=(longEW == true ? -longitude : longitude);

    		boolean latSN=(a2 & (1 << 0)) != 0 ? true : false;
    		char NS=(latSN == true ? 'N' : 'S');
    		Print.logInfo(""+NS);
    		//double NS1=(latSN == true ? latitude : -latitude);
    		//Print.logInfo(""+NS1);
    		latitude=(latSN == true ? latitude : -latitude);

    		Print.logInfo("ID Tracker: " + gt06DeviceID);

    		/* GPS Event */
            this.gpsEvent = this.createGPSEvent(gt06DeviceID);//modemID);
            if (this.gpsEvent == null) {
                // errors already displayed
                return null;
            }
        	
        	Device device = this.gpsEvent.getDevice();
        	if (device == null) {
        		return null;
        	}
        	String accountID = device.getAccountID();
            String deviceID  = device.getDeviceID();
            String uniqueID  = device.getUniqueID();
            System.out.println("Registro ---> [" + accountID + "/" + deviceID + "/" + uniqueID + "]");
            System.out.println("Evento: [" + new DateTime(fixtime) + "]");
        	
        	if ((speedKPH >= 5) && (speedKPH <= 90)) {
        		statusCode = StatusCodes.STATUS_MOTION_IN_MOTION;
        	} else if (speedKPH >= 91) {
        		statusCode = StatusCodes.STATUS_MOTION_EXCESS_SPEED;
        	} else if (speedKPH <= 4) {
        		statusCode = StatusCodes.STATUS_MOTION_STOP;
        	}
        	if (powerCut){
        		statusCode 	= StatusCodes.STATUS_POWER_FAILURE;
        	}
        	
        	if (ignition){
        		statusCode 	= StatusCodes.STATUS_IGNITION_ON;
        	}else{
        		statusCode 	= StatusCodes.STATUS_IGNITION_OFF;
        	}
        	if (panic){
        		statusCode  = StatusCodes.STATUS_PANIC_ON;
        	}

        	/* Alerta de Velocidad MTC - SUTRAN */
        	speedKPH = this.gpsEvent.ajustaVelocidad(device.getDescription(), speedKPH);
        	this.lastModemID 	=	gt06DeviceID;//modemID;

        	this.gpsEvent.setTimestamp(fixtime);
        	this.gpsEvent.setStatusCode(statusCode);
        	this.gpsEvent.setLatitude(latitude);
        	this.gpsEvent.setLongitude(longitude);
        	this.gpsEvent.setSpeedKPH(speedKPH);
        	this.gpsEvent.setHeading(heading);
        	this.gpsEvent.setSatelliteCount(satsCount);
        	this.gpsEvent.setBatteryLevel(voltLevel);
        	//this.gpsEvent.setAltitude(altitudeM);
        	this.gpsEvent.setSignalStrength(signalRSSI);
        	
        	/* insert/return */
        	if (this.parseInsertRecord_Common(this.gpsEvent)) {
        		// change this to return any required acknowledgement (ACK) packets back to the device
        		return null;
        	} else {
        		return null;
        	}
    	}
    	
	}
    
    /* parse and insert data record (common) */
    private boolean parseInsertRecord_Common(GPSEvent gpsEv)
    {
        long   fixtime    = gpsEv.getTimestamp();
        int    statusCode = gpsEv.getStatusCode();
        Device dev        = gpsEv.getDevice(); // guaranteed non-null here

        /* invalid date? */
        if (fixtime <= 0L) {
            Print.logWarn("Invalid date/time");
            fixtime = DateTime.getCurrentTimeSec(); // default to now
            gpsEv.setTimestamp(fixtime);
        }
                
        /* valid lat/lon? */
        if (!gpsEv.isValidGeoPoint()) {
            Print.logWarn("Invalid lat/lon: " + gpsEv.getLatitude() + "/" + gpsEv.getLongitude());
            gpsEv.setLatitude(0.0);
            gpsEv.setLongitude(0.0);
        }
        GeoPoint geoPoint = gpsEv.getGeoPoint();

        /* minimum speed */
        if (gpsEv.getSpeedKPH() < MINIMUM_SPEED_KPH) {
            gpsEv.setSpeedKPH(0.0);
            gpsEv.setHeading(0.0);
        }

        /* estimate GPS-based odometer */
        double odomKM = 0.0; // set to available odometer from event record
        //if (this.gpsDevice.getLastEventTimestamp() < fixtime) {
        if (odomKM <= 0.0) {
            odomKM = (ESTIMATE_ODOMETER && geoPoint.isValid())? 
                this.gpsDevice.getNextOdometerKM(geoPoint) : 
                this.gpsDevice.getLastOdometerKM();
        } else {
            odomKM = dev.adjustOdometerKM(odomKM);
        }
        //}
        Print.logInfo("Odometer KM: " + odomKM);
        gpsEv.setOdometerKM(odomKM);

        /* simulate Geozone arrival/departure */
        if (SIMEVENT_GEOZONES && geoPoint.isValid()) {
            java.util.List<Device.GeozoneTransition> zone = dev.checkGeozoneTransitions(fixtime, geoPoint);
            if (zone != null) {
                for (Device.GeozoneTransition z : zone) {
                    gpsEv.insertEventData(z.getTimestamp(), z.getStatusCode(), z.getGeozone());
                    Print.logInfo("Geozone    : " + z);
                }
            }
        }

        /* digital input change events */
        if (gpsEv.hasInputMask() && (gpsEv.getInputMask() >= 0L)) {
            long gpioInput = gpsEv.getInputMask();
            if (SIMEVENT_DIGITAL_INPUTS > 0L) {
                // The current input state is compared to the last value stored in the Device record.
                // Changes in the input state will generate a synthesized event.
                long chgMask = (dev.getLastInputState() ^ gpioInput) & SIMEVENT_DIGITAL_INPUTS;
                if (chgMask != 0L) {
                    // an input state has changed
                    for (int b = 0; b <= 15; b++) {
                        long m = 1L << b;
                        if ((chgMask & m) != 0L) {
                            // this bit changed
                            int  inpCode = ((gpioInput & m) != 0L)? InputStatusCodes_ON[b] : InputStatusCodes_OFF[b];
                            long inpTime = fixtime;
                            gpsEv.insertEventData(inpTime, inpCode);
                            Print.logInfo("GPIO : " + StatusCodes.GetDescription(inpCode,null));
                        }
                    }
                }
            }
            dev.setLastInputState(gpioInput & 0xFFFFL); // FLD_lastInputState
        }

        /* create/insert standard event */
        gpsEv.insertEventData(fixtime, statusCode);

        /* save Device changes */
        gpsEv.updateDevice();

        /* return success */
        return true;

    }

    /**
    *** Initialize runtime configuration
    **/
    public static void configInit() 
    {
        DCServerConfig dcsc     = Main.getServerConfig(null);
        if (dcsc == null) {
            Print.logWarn("DCServer not found: " + Main.getServerName());
            return;
        }

        /* custom */
        DATA_FORMAT_OPTION      = dcsc.getIntProperty(Main.ARG_FORMAT, DATA_FORMAT_OPTION);

        /* common */
        MINIMUM_SPEED_KPH       = dcsc.getMinimumSpeedKPH(MINIMUM_SPEED_KPH);
        ESTIMATE_ODOMETER       = dcsc.getEstimateOdometer(ESTIMATE_ODOMETER);
        SIMEVENT_GEOZONES       = dcsc.getSimulateGeozones(SIMEVENT_GEOZONES);
        SIMEVENT_DIGITAL_INPUTS = dcsc.getSimulateDigitalInputs(SIMEVENT_DIGITAL_INPUTS) & 0xFFFFL;

    }

    // ------------------------------------------------------------------------
    // ------------------------------------------------------------------------
    // ------------------------------------------------------------------------
    // Once you have modified this example 'template' server to parse your particular
    // Device packets, you can also use this source module to load GPS data packets
    // which have been saved in a file.  To run this module to load your save GPS data
    // packets, start this command as follows:
    //   java -cp <classpath> org.opengts.servers.template.TrackClientPacketHandler {options}
    // Where your options are one or more of 
    //   -insert=[true|false]    Insert parse records into EventData
    //   -format=[1|2]           Data format
    //   -debug                  Parse internal sample data
    //   -parseFile=<file>       Parse data from specified file

    private static int _usage()
    {
        String cn = StringTools.className(TrackClientPacketHandler.class);
        Print.sysPrintln("Test/Load Device Communication Server");
        Print.sysPrintln("Usage:");
        Print.sysPrintln("  $JAVA_HOME/bin/java -classpath <classpath> %s {options}", cn);
        Print.sysPrintln("Options:");
        Print.sysPrintln("  -insert=[true|false]    Insert parsed records into EventData");
        Print.sysPrintln("  -format=[1|2]           Data format");
        Print.sysPrintln("  -debug                  Parse internal sample/debug data (if any)");
        Print.sysPrintln("  -parseFile=<file>       Parse data from specified file");
        return 1;
    }

    /**
    *** Main entry point used for debug purposes only
    **/
    public static int _main(boolean fromMain)
    {

        /* default options */
        INSERT_EVENT = RTConfig.getBoolean(Main.ARG_INSERT, DFT_INSERT_EVENT);
        if (!INSERT_EVENT) {
            Print.sysPrintln("Warning: Data will NOT be inserted into the database");
        }

        /* create client packet handler */
        TrackClientPacketHandler tcph = new TrackClientPacketHandler();

        /* DEBUG sample data */
        if (RTConfig.getBoolean(Main.ARG_DEBUG,false)) {
            String data[] = null;
            switch (DATA_FORMAT_OPTION) {
                case  1: data = new String[] {
                    "123456789012345,2006/09/05,07:47:26,35.3640,-142.2958,27.0,224.8",
                }; break;
                case  2: data = new String[] {
                    "account/device/$GPRMC,025423.494,A,3709.0642,N,14207.8315,W,12.09,108.52,200505,,*2E",
                    "/device/$GPRMC,025423.494,A,3709.0642,N,14207.8315,W,12.09,108.52,200505,,*2E",
                }; break;
                case  3: data = new String[] {
                    "2,123,1234567890,0,20101223,110819,1,2.1,39.1234,-142.1234,33,227,1800",
                }; break;
                case  9: data = new String[] {
                    "mid=123456789012345 lat=39.12345 lon=-142.12345 kph=123.0"
                }; break;
                default:
                    Print.sysPrintln("Unrecognized Data Format: %d", DATA_FORMAT_OPTION);
                    return _usage();
            }
            for (int i = 0; i < data.length; i++) {
                tcph.getHandlePacket(data[i].getBytes());
            }
            return 0;
        }

        /* 'parseFile' specified? */
        if (RTConfig.hasProperty(Main.ARG_PARSEFILE)) {

            /* get input file */
            File parseFile = RTConfig.getFile(Main.ARG_PARSEFILE,null);
            if ((parseFile == null) || !parseFile.isFile()) {
                Print.sysPrintln("Data source file not specified, or does not exist.");
                return _usage();
            }

            /* open file */
            FileInputStream fis = null;
            try {
                fis = new FileInputStream(parseFile);
            } catch (IOException ioe) {
                Print.logException("Error openning input file: " + parseFile, ioe);
                return 2;
            }

            /* loop through file */
            try {
                // records are assumed to be terminated by CR/NL 
                for (;;) {
                    String data = FileTools.readLine(fis);
                    if (!StringTools.isBlank(data)) {
                        tcph.getHandlePacket(data.getBytes());
                    }
                }
            } catch (EOFException eof) {
                Print.sysPrintln("");
                Print.sysPrintln("***** End-Of-File *****");
            } catch (IOException ioe) {
                Print.logException("Error reaading input file: " + parseFile, ioe);
            } finally {
                try { fis.close(); } catch (Throwable th) {/* ignore */}
            }

            /* done */
            return 0;

        }

        /* no options? */
        return _usage();

    }

    /**
    *** Main entry point used for debug purposes only
    **/
    public static void main(String argv[])
    {
        DBConfig.cmdLineInit(argv,false);
        TrackClientPacketHandler.configInit();
        System.exit(TrackClientPacketHandler._main(false));
    }
    
}
