package org.opengts.servers.gl100;

import java.util.*;
import java.io.*;
import java.net.*;
import org.opengts.util.*;
import org.opengts.db.*;
import org.opengts.db.tables.*;

import org.opengts.servers.*;

/**
*** <code>TrackClientPacketHandler</code> - This module contains the general
*** "business logic" for parsing incoming data packets from the remote tracking
*** Device.
**/

public class TrackClientPacketHandler
    extends AbstractClientPacketHandler
{
    public  static 		int	DATA_FORMAT_OPTION				= 1;

    //    vehicle value.
    public  static		boolean ESTIMATE_ODOMETER          = true;
    
    /* simulate geozone arrival/departure */
    // (enable to insert simulated Geozone arrival/departure EventData records)
    public  static       boolean SIMEVENT_GEOZONES          = true;
    
    /* simulate digital input changes */
    public  static       long    SIMEVENT_DIGITAL_INPUTS    = 0x0000L; // 0xFFFFL;

    /* flag indicating whether data should be inserted into the DB */
    // should be set to 'true' for production.
    private static       boolean DFT_INSERT_EVENT           = true;
    private static       boolean INSERT_EVENT               = DFT_INSERT_EVENT;

    /* update Device record */
    // (enable to update Device record with current IP address and last connect time)
  //private static       boolean UPDATE_DEVICE              = false;
    
    /* minimum acceptable speed value */
    // Speeds below this value should be considered 'stopped'
    public  static       double  MINIMUM_SPEED_KPH          = 0.0;

    /* Knot/Kilometer conversions */
    public static final double  KILOMETERS_PER_KNOT     = 1.85200000;
    public static final double  KNOTS_PER_KILOMETER     = 1.0 / KILOMETERS_PER_KNOT;

    // ------------------------------------------------------------------------

    /* Ingore $GPRMC checksum? */
    // (only applicable for data formats that include NMEA-0183 formatted event records)
    //private static       boolean IGNORE_NMEA_CHECKSUM       = false;

    // ------------------------------------------------------------------------

    /* GMT/UTC timezone */
    private static final TimeZone gmtTimezone               = DateTime.getGMTTimeZone();

    // ------------------------------------------------------------------------
    
    private 			String 		tipoData				=	"";

    /* GTS status codes for Input-On events */
    private static final int InputStatusCodes_ON[] = new int[] {
        StatusCodes.STATUS_INPUT_ON_00,
        StatusCodes.STATUS_INPUT_ON_01,
        StatusCodes.STATUS_INPUT_ON_02,
        StatusCodes.STATUS_INPUT_ON_03,
        StatusCodes.STATUS_INPUT_ON_04,
        StatusCodes.STATUS_INPUT_ON_05,
        StatusCodes.STATUS_INPUT_ON_06,
        StatusCodes.STATUS_INPUT_ON_07,
        StatusCodes.STATUS_INPUT_ON_08,
        StatusCodes.STATUS_INPUT_ON_09,
        StatusCodes.STATUS_INPUT_ON_10,
        StatusCodes.STATUS_INPUT_ON_11,
        StatusCodes.STATUS_INPUT_ON_12,
        StatusCodes.STATUS_INPUT_ON_13,
        StatusCodes.STATUS_INPUT_ON_14,
        StatusCodes.STATUS_INPUT_ON_15
    };

    /* GTS status codes for Input-Off events */
    private static final int InputStatusCodes_OFF[] = new int[] {
        StatusCodes.STATUS_INPUT_OFF_00,
        StatusCodes.STATUS_INPUT_OFF_01,
        StatusCodes.STATUS_INPUT_OFF_02,
        StatusCodes.STATUS_INPUT_OFF_03,
        StatusCodes.STATUS_INPUT_OFF_04,
        StatusCodes.STATUS_INPUT_OFF_05,
        StatusCodes.STATUS_INPUT_OFF_06,
        StatusCodes.STATUS_INPUT_OFF_07,
        StatusCodes.STATUS_INPUT_OFF_08,
        StatusCodes.STATUS_INPUT_OFF_09,
        StatusCodes.STATUS_INPUT_OFF_10,
        StatusCodes.STATUS_INPUT_OFF_11,
        StatusCodes.STATUS_INPUT_OFF_12,
        StatusCodes.STATUS_INPUT_OFF_13,
        StatusCodes.STATUS_INPUT_OFF_14,
        StatusCodes.STATUS_INPUT_OFF_15
    };

    // ------------------------------------------------------------------------
    // ------------------------------------------------------------------------

    /* TCP session ID */
    private String          sessionID                   = null;

    /* common GPSEvent instance */
    private GPSEvent        gpsEvent                    = null;

    /* Device record */
    private Device          gpsDevice                   = null;
    private String          lastModemID                 = null;

    /* Session 'terminate' indicator */
    // This value should be set to 'true' when this server has determined that the
    // session should be terminated.  For instance, if this server finishes communication
    // with the Device or if parser finds a fatal error in the incoming data stream 
    // (ie. invalid Account/Device, or unrecognizable data).
    private boolean         terminate                   = false;

    /* session IP address */
    // These values will be set for you by the incoming session to indicate the 
    // originating IP address.
    private String          ipAddress                   = null;
    private int             clientPort                  = 0;

    /* packet handler constructor */
    public TrackClientPacketHandler() 
    {
        super();
        //Print.logStackTrace("new TrackClientPacketHandler ...");
    }

    // ------------------------------------------------------------------------

    /* callback when session is starting */
    // this method is called at the beginning of a communication session
    public void sessionStarted(InetAddress inetAddr, boolean isTCP, boolean isText)
    {
        super.sessionStarted(inetAddr, isTCP, isText);
        super.clearTerminateSession();

        /* init */
        this.ipAddress        = (inetAddr != null)? inetAddr.getHostAddress() : null;
        this.clientPort       = this.getSessionInfo().getRemotePort();
        if (this.isDuplex()) {
        	this.tipoData="TCP";
            //Print.logInfo("Inicio TCP: " + this.ipAddress + " [" + new DateTime() + "]");
        	//	Print.logInfo("========================================================");
        } else {
        	this.tipoData="UDP";
        	//Print.logInfo("Inicio UDP: " + this.ipAddress + " [" + new DateTime() + "]");
        	//Print.logInfo("========================================================");
        }

    }

    /* callback when session is terminating */
    // this method is called at the end of a communication session
    public void sessionTerminated(Throwable err, long readCount, long writeCount)
    {
        super.sessionTerminated(err, readCount, writeCount);
    }

    /* callback to return the TCP session id */
    public String getSessionID()
    {
        if (!StringTools.isBlank(this.sessionID)) {
            return this.sessionID;
        } else
        if (this.gpsDevice != null) {
            return CreateTcpSessionID(this.gpsDevice);
        } else
        if (this.gpsEvent != null) {
            return CreateTcpSessionID(this.gpsEvent.getDevice());
        } else {
            return null;
        }
    }

    // ------------------------------------------------------------------------

    public static final boolean USE_STANDARD_TCP_SESSION_ID = true;

    /* get TCP session ID */
    public static String GetTcpSessionID(Device dev)
    {
        if (USE_STANDARD_TCP_SESSION_ID || (dev == null)) {
            return DCServerFactory.getTcpSessionID(dev);
        } else {
            // Custom example for extracting the session ID from the Device 
            // UniqueID (instead of the DeviceID)
            String uid = dev.getUniqueID();
            int    u   = uid.lastIndexOf("_");
            if (u < 0) {
                // no "_" character, fallback to standard TCP sessionID
                return DCServerFactory.getTcpSessionID(dev);
            } else {
                String sessID = dev.getAccountID() + "/" + uid.substring(0,u+1);
                return sessID;
            }
        }
    }

    /* create TCP session ID */
    public static String CreateTcpSessionID(Device dev)
    {
        if (USE_STANDARD_TCP_SESSION_ID || (dev == null)) {
            return DCServerFactory.createTcpSessionID(dev);
        } else {
            String uid = dev.getUniqueID();
            int    u   = uid.lastIndexOf("_");
            if (u < 0) {
                // no "_" character, fallback to standard TCP sessionID
                return DCServerFactory.createTcpSessionID(dev);
            } else {
                String sessID = dev.getAccountID() + "/" + uid.substring(0,u+1);
                dev.setLastTcpSessionID(sessID);
                return sessID;
            }
        }
    }

     /**
     *** Create GPSEvent from the ModemID
     *** @return The created GPSEvent instance, or null if the ModemID is invalid
     **/
     private GPSEvent createGPSEvent(String modemID)
     {
         DCServerConfig dcserver = Main.getServerConfig(null);

         /* create/load device */
         if (this.gpsDevice != null) {
             if (!modemID.equals("*") && 
                 ((this.lastModemID == null) || !this.lastModemID.equals(modemID))) { // fix [B10]
                 Print.logError("New MobileID does not match previously loaded Device");
                 //return this.gpsEvent;
                 return null;
             }
             this.gpsEvent  = new GPSEvent(dcserver, this.ipAddress, this.clientPort, this.gpsDevice);
         } else {
             if (StringTools.isBlank(modemID) || modemID.equals("*")) {
                 Print.logWarn("ModemID not specified!");
                 return null;
             }
             this.gpsEvent  = new GPSEvent(dcserver, this.ipAddress, this.clientPort, modemID);
             this.gpsDevice = this.gpsEvent.getDevice(); // may still be null
         }

         /* no Device? */
         if (this.gpsDevice == null) {
             // errors already displayed
             return null;
         }

         /* create session ID */
         this.sessionID = CreateTcpSessionID(this.gpsDevice);

         /* return GPSEvent */
         return this.gpsEvent; // non-null

     }

    /* based on the supplied packet data, return the remaining bytes to read in the packet */
    public int getActualPacketLength(byte packet[], int packetLen)
    {
        if (Constants.ASCII_PACKETS) {
            // (this actually won't be called if 'Constants.ASCII_PACKETS' is true).
            // ASCII packets - look for line terminator [see Constants.ASCII_LINE_TERMINATOR)]
            return ServerSocketThread.PACKET_LEN_LINE_TERMINATOR;  // read until line termination character
            //return ServerSocketThread.PACKET_LEN_END_OF_STREAM;  // read until end of stream, or maxlen
        } else {
            // BINARY packet - need to analyze 'packet[]' and determine actual packet length
            //return ServerSocketThread.PACKET_LEN_LINE_TERMINATOR; // <-- change this for binary packets
        	return ServerSocketThread.PACKET_LEN_END_OF_STREAM; // <-- change this for binary packets
        }
        
    }

    /* set session terminate after next packet handling */
    private void setTerminate()
    {
        this.terminate = true;
    }
    
    /* indicate that the session should terminate */
    // This method is called after each return from "getHandlePacket" to check to see
    // the current session should be closed.
    public boolean getTerminateSession()
    {
        return this.terminate;
    }

    // ------------------------------------------------------------------------
    // ------------------------------------------------------------------------

    /* return the initial packet sent to the Device after session is open */
    public byte[] getInitialPacket() 
        throws Exception
    {
        // At this point a connection from the client to the server has just been
        // initiated, and we have not yet received any data from the client.
        // If the client is expecting to receive an initial packet from the server at
        // the time that the client connects, then this is where the server can return
        // a byte array that will be transmitted to the client Device.
        return null;
        // Note: any returned response for "getInitialPacket()" is ignored for simplex/udp connections.
        // Returned UDP packets may be sent from "getHandlePacket" or "getFinalPacket".
    }

    /* workhorse of the packet handler */
    public byte[] getHandlePacket(byte pktBytes[]) 
    {
        if ((pktBytes != null) && (pktBytes.length > 0)) {
            
        	String s = StringTools.toStringValue(pktBytes).trim(); // remove leading/trailing spaces
            
            // $DEV,A,DATE,TIME,LAT,LON,SPEED,ANGLE,ALTITUD,F422,BLANCO,NARANJA,AMARILLO,SAL_MARRON,SAL_VERDE,ADC,BATT,CSQ,TEMP*
            // $$a7q969,A,230411,112030,-1203.1230,-77.91823,12,13,300,F422,1,1,1,0,0,165,22,33*
    		// +RESP:GTTRI,352134010770825,3,0,0,1,0.0,181,381.0,2,-112.057885,33.630660,20081120223023
            byte rtn[] = null;
            if (s.startsWith("+RESP")){
            	Print.logInfo("------ Trama GL100 -----");
            	rtn = this.parseInsertRecordGL100(s);
            } else if (s.startsWith("$")){
            	Print.logInfo("------ Trama Raptor -----");
            	rtn = this.parseInsertRecordRaptor(s);
            } else if (s.startsWith(">")){
            	Print.logInfo("------ Trama Alave -----");
            	rtn = this.parseInsertRecordAlave(s);
            } else if (s.startsWith("link")){
            	Print.logInfo("MTC UPDATER: " + s);
            	rtn = this.parseInsertRecordUpdater(s);
            } else {
            	Print.logInfo("No Conozco esta trama");
            }
        	
            /* return response */
            return rtn; // no return packets are expected

        } else {

            /* no packet date received */
            Print.logInfo("...");
            return null; // no return packets are expected

        }
    }

    /* final packet sent to Device before session is closed */
    public byte[] getFinalPacket(boolean hasError) 
        throws Exception
    {
        return null;
    }
    
    /* parse the specified date into unix 'epoch' time */
    private long _parsearFechas(String ymd, String hms)
    {
        // "YYYY/MM/DD", "hh:mm:ss"
	  String d[] = new String[3];
		d[0] = ymd.substring(0,4);
		d[1] = ymd.substring(4,6);
		d[2] = ymd.substring(6,8);
	  String t[] = new String[3];
		t[0] = hms.substring(0,2);
		t[1] = hms.substring(2,4);
		t[2] = hms.substring(4,6);

	  if ((d.length != 3) && (t.length != 3)) {
            //Print.logError("Invalid date: " + ymd + ", " + hms);
            return 0L;
        } else {
            int YY = StringTools.parseInt(d[0],0); // 07 year
            int MM = StringTools.parseInt(d[1],0); // 04 month
            int DD = StringTools.parseInt(d[2],0); // 18 day
            int hh = StringTools.parseInt(t[0],0); // 01 hour
            int mm = StringTools.parseInt(t[1],0); // 48 minute
            int ss = StringTools.parseInt(t[2],0); // 04 second
            if (YY < 100) { YY += 2000; }
            DateTime dt = new DateTime(gmtTimezone,YY,MM,DD,hh,mm,ss);
            return dt.getTimeSec();
        }
    }
    
    private long _parseDate7(String ymd, String hms)
    {
        // "230411", "hh:mm:ss"
	  String d[] = new String[3];
		d[0] = ymd.substring(4,6);
		d[1] = ymd.substring(2,4);
		d[2] = ymd.substring(0,2);
	  String t[] = new String[3];
		t[0] = hms.substring(0,2);
		t[1] = hms.substring(2,4);
		t[2] = hms.substring(4,6);

	  if ((d.length != 3) && (t.length != 3)) {
            //Print.logError("Invalid date: " + ymd + ", " + hms);
            return 0L;
        } else {
            int YY = StringTools.parseInt(d[0],0); // 07 year
            int MM = StringTools.parseInt(d[1],0); // 04 month
            int DD = StringTools.parseInt(d[2],0); // 18 day
            int hh = StringTools.parseInt(t[0],0); // 01 hour
            int mm = StringTools.parseInt(t[1],0); // 48 minute
            int ss = StringTools.parseInt(t[2],0); // 04 second
            if (YY < 100) { YY += 2000; }
            DateTime dt = new DateTime(gmtTimezone,YY,MM,DD,hh,mm,ss);
            return dt.getTimeSec();
        }
    }
    
    private double _buscaLatitud(String s, String d)
	{
     
    	double _lat = StringTools.parseDouble(s, 99999.0);
    	if (_lat < 99999.0) {
    		double lat = (double)((long)_lat / 100L); // _lat is always positive here
            lat += (_lat - (lat * 100.0)) / 60.0;
            return d.equals("S")? -lat : lat;
            //return lat;
        } else {
  	      return 90.0; // invalid latitude
        }
	}

    private double _buscaLongitud(String s, String d)
    {
    	double _lon = StringTools.parseDouble(s, 99999.0);
    	if (_lon < 99999.0) {
    		double lon = (double)((long)_lon / 100L); // _lon is always positive here
    		lon += (_lon - (lon * 100.0)) / 60.0;
    		return d.equals("W")? -lon : lon;
    		//return lon;
    	} else {
    		return 180.0; // invalid longitude
    	}
    }
    
    /* parse and insert data record */
    private byte[] parseInsertRecordRaptor(String s)
    {
    	Print.logInfo("Raptor: " +s);
    	String  fld[] = StringTools.parseStringArray(s, ',');
    	if (fld == null) {
    		Print.logWarn("Fields are null");
    		return null;
    	}
    	
    	String	modemID=fld[0].toLowerCase().substring(1, fld[0].length());
    	String  dateRaptor=fld[2].toLowerCase();
    	String  timeRaptor=fld[3].toLowerCase();
    	long    fixtime=this._parseDate7(dateRaptor,timeRaptor);
    	double latitudDec	= _buscaLatitud(fld[4].substring(0,fld[4].length() - 1 ), fld[4].substring(fld[4].length() -1 ,fld[4].length()));
    	double longitudDec = _buscaLongitud(fld[5].substring(0,fld[5].length() - 1 ), fld[5].substring(fld[5].length() -1 ,fld[5].length()));
    	double latitude   	= StringTools.parseDouble(latitudDec,0.0);
    	double longitude 	= StringTools.parseDouble(longitudDec,0.0);
    	double speedKPH  	= StringTools.parseDouble(fld[6],0.0);
    	double heading		= StringTools.parseDouble(fld[7],0.0);
    	double altitudeM = StringTools.parseDouble(fld[8],0.0);
    	
    	String hex = fld[9];
    	int statusCode=Integer.parseInt(hex, 16);
    	
    	if ((latitude >=-1 || latitude <=-20) || (longitude >=-68 || longitude <= -82))
    	{
    		System.out.println("Error en Posicion");
    		return null;
    	}
    	/* Bandera, Sensores 1+2+3, Salidas 1+2 */
    	String	Sen1		= fld[10];
    	String	Door		= fld[11];
    	String	Sen2		= fld[12];
    	String 	Sal1		= fld[13];
    	String  Sal2		= fld[14];
    	String entradas = (Sen1 + Door + Sen2 + Sal1 + Sal2);
    	long inputs = StringTools.parseLong(entradas, 0);

    	/* ADC, ADC, Batt, CSQ, CMTEE */
    	String	Str_adc		= fld[15].substring(0,fld[15].length() -1);
    	double  adcValue	= StringTools.parseDouble(Str_adc, 0.0);
    	double 	fuelLevel 	= ((adcValue * 100) / 165);
    	fuelLevel = fuelLevel - 100;
    	fuelLevel = -fuelLevel;

    	/* GPS Event */
        this.gpsEvent = this.createGPSEvent(modemID);
        if (this.gpsEvent == null) {
            // errors already displayed
            return null;
        }
    	
    	Device device = this.gpsEvent.getDevice();
    	if (device == null) {
    		return null;
    	}
    	String accountID = device.getAccountID();
        String deviceID  = device.getDeviceID();
        String uniqueID  = device.getUniqueID();
        System.out.println("Registro ---> [" + accountID + "/" + deviceID + "/" + uniqueID + "]");
        System.out.println("Evento: [" + new DateTime(fixtime) + "]");
    	
        /** Ajuste de Velocidad - Motion Start **/
        double maxSpeed	= device.getSpeedLimitKPH();
        if ((speedKPH > MINIMUM_SPEED_KPH) && (speedKPH < maxSpeed)) {
        	if(device.getLastValidSpeed() <= MINIMUM_SPEED_KPH){
        		statusCode = StatusCodes.STATUS_MOTION_START;
        	}else{
        		statusCode = StatusCodes.STATUS_MOTION_IN_MOTION;	 
        	}

        } else if (speedKPH >= maxSpeed) {
        	statusCode = StatusCodes.STATUS_MOTION_EXCESS_SPEED;
        } else if (speedKPH <= MINIMUM_SPEED_KPH) {
        	statusCode = StatusCodes.STATUS_MOTION_STOP;
        }

    	/* Alerta de Velocidad MTC - SUTRAN */
    	speedKPH = this.gpsEvent.ajustaVelocidad(device.getDescription(), speedKPH);
    	this.lastModemID 	=	modemID;
    	
    	this.gpsEvent.setTimestamp(fixtime);
    	this.gpsEvent.setStatusCode(statusCode);
    	this.gpsEvent.setLatitude(latitude);
    	this.gpsEvent.setLongitude(longitude);
    	this.gpsEvent.setSpeedKPH(speedKPH);
    	this.gpsEvent.setHeading(heading);
    	this.gpsEvent.setBatteryLevel(100);
    	this.gpsEvent.setAltitude(altitudeM);
    	this.gpsEvent.setInputMask(inputs);
    	this.gpsEvent.setFuelTotal(fuelLevel);
    	
    	/* insert/return */
    	if (this.parseInsertRecord_Common(this.gpsEvent)) {
    		// change this to return any required acknowledgement (ACK) packets back to the device
    		return null;
    	} else {
    		return null;
    	}
    	
    }
    
    private byte[] parseInsertRecordGL100(String s)
    {
    	// +RESP:GTTRI,352134010770825,3,0,0,1,0.0,181,381.0,2,-112.057885,33.630660,20081120223023
		// +RESP:GTTRI,011874000100466,1,0,0,1,0.0,11,207.0,3,-77.025563,-12.055870,20100906162025,0716,0010,0463,1039,00,0007,0102100203
		// +RESP:GTSZI |               | | | | |   |   |     | |           |         |>Time Stamp YYYYMMDDHHSS [12]
		// +RESP:GTSOS |               | | | | |   |   |     | |           |>Latitude [11]
		// +RESP:GTRTL |               | | | | |   |   |     | |>Longitud [10]
		//             |               | | | | |   |   |     |>Precision de Posicion (1-50, menos de 15 es bueno. si GPS FIX=1)  [9]
		//             |               | | | | |   |   |>Altitud [8]
		//             |               | | | | |   |>Curso [7]
		//             |               | | | | |>Vel [6]
		//             |               | | | |>GPS Fix; 1=fix, 0=no fix [5]
		//             |               | | |>Alerta Zona; 1=Entra, 0=Sale (GTSZI solamente) [4]
		//             |               | |>Zona ID; 0-4 (GTSZI solamente) [3]
		//             |               |>Numero de Reportes (GTTRI solamente) [2]
		//             |>IMEI [1]
		//
    	Print.logInfo("GL100: " +s);
    	String  fld[] = StringTools.parseStringArray(s, ',');
    	if (fld == null) {
    		Print.logWarn("Fields are null");
    		return null;
    	}
    	
    	if ((fld[0].contains("GTTRI")) || (fld[0].contains("GTRTL")) || (fld[0].contains("GTSZI")))
    	{
    		System.out.println("Reporte OK!");
    		// +RESP:GTTRI,352134010770825,3,0,0,1,0.0,181,381.0,2,-112.057885,33.630660,20081120223023
    	}

    	//String	activeFix	=	fld[5];
    	String	modemID		=	fld[1];
    	String  dateGL100	=	fld[12].toLowerCase().substring(0, 8);
    	String  timeGL100	=	fld[12].toLowerCase().substring(8, 14);
    	long    fixtime		=	this._parsearFechas(dateGL100,timeGL100);
    	int		statusCode	= StatusCodes.STATUS_LOCATION;
    	double latitude   	= StringTools.parseDouble(fld[11],0.0);
    	double longitude 	= StringTools.parseDouble(fld[10],0.0);
    	double speedKPH  	= StringTools.parseDouble(fld[6],0.0);
    	double heading		= StringTools.parseDouble(fld[7],0.0);
    	double altitudeM	= StringTools.parseDouble(fld[8],0.0);
    	
    	/* GPS Event */
        this.gpsEvent = this.createGPSEvent(modemID);
        if (this.gpsEvent == null) {
            // errors already displayed
            return null;
        }
    	
    	Device device = this.gpsEvent.getDevice();
    	if (device == null) {
    		return null;
    	}
    	String accountID = device.getAccountID();
        String deviceID  = device.getDeviceID();
        String uniqueID  = device.getUniqueID();
        System.out.println("Registro ---> [" + accountID + "/" + deviceID + "/" + uniqueID + "]");
        System.out.println("Evento: [" + new DateTime(fixtime) + "]");
    	
        /** Ajuste de Velocidad - Motion Start **/
        double maxSpeed	= device.getSpeedLimitKPH();
        if ((speedKPH > MINIMUM_SPEED_KPH) && (speedKPH < maxSpeed)) {
        	if(device.getLastValidSpeed() <= MINIMUM_SPEED_KPH){
        		statusCode = StatusCodes.STATUS_MOTION_START;
        	}else{
        		statusCode = StatusCodes.STATUS_MOTION_IN_MOTION;	 
        	}

        } else if (speedKPH >= maxSpeed) {
        	statusCode = StatusCodes.STATUS_MOTION_EXCESS_SPEED;
        } else if (speedKPH <= MINIMUM_SPEED_KPH) {
        	statusCode = StatusCodes.STATUS_MOTION_STOP;
        }

    	/* Alerta de Velocidad MTC - SUTRAN */
    	speedKPH = this.gpsEvent.ajustaVelocidad(device.getDescription(), speedKPH);
    	this.lastModemID 	=	modemID;
    	
    	this.gpsEvent.setTimestamp(fixtime);
    	this.gpsEvent.setStatusCode(statusCode);
    	this.gpsEvent.setLatitude(latitude);
    	this.gpsEvent.setLongitude(longitude);
    	this.gpsEvent.setSpeedKPH(speedKPH);
    	this.gpsEvent.setHeading(heading);
    	this.gpsEvent.setBatteryLevel(100);
    	this.gpsEvent.setAltitude(altitudeM);
    	
    	/* insert/return */
    	if (this.parseInsertRecord_Common(this.gpsEvent)) {
    		// change this to return any required acknowledgement (ACK) packets back to the device
    		return null;
    	} else {
    		return null;
    	}

    }
    
    private byte[] parseInsertRecordAlave(String s)
    {
    	// >id,cellphone<
    	Print.logInfo("Alave: " +s);
    	String  fld[] = StringTools.parseStringArray(s, ',');
    	if (fld == null) {
    		Print.logWarn("Fields are null");
    		return null;
    	}
    	
    	String modemID = fld[0].substring(1);
    	long fixtime = (new DateTime(DateTime.getGMTTimeZone())).getTimeSec();//getDayStart() + gpsTOD;
        if ((fixtime - DateTime.MinuteSeconds(15)) > DateTime.getCurrentTimeSec()) {
            fixtime -= DateTime.DaySeconds(1);
        }
    	int		statusCode	= StatusCodes.STATUS_LOCATION;
    	double latitude   	= 0.0;
    	double longitude 	= 0.0;
    	double speedKPH  	= 0.0;
    	double heading	= 0.0;
    	double altitudeM	= 0.0;
    	String flag			= fld[2];
    	/* GPS Event */
        this.gpsEvent = this.createGPSEvent(modemID);
        if (this.gpsEvent == null) {
            // errors already displayed
            return null;
        }
    	
    	Device device = this.gpsEvent.getDevice();
    	if (device == null) {
    		return null;
    	}
    	
    	latitude = StringTools.parseDouble(device.getFixedLatitude(), 0.0);
    	longitude = StringTools.parseDouble(device.getFixedLongitude(), 0.0);
    	
    	if (!GeoPoint.isValid(latitude,longitude)) {
            Print.logWarn("Invalid lat/lon: " + latitude + "/" + longitude);
            latitude   = 0.0;
            longitude  = 0.0;
            speedKPH   = 0.0;
        }
        
        //GeoPoint geoPoint = new GeoPoint(latitude,longitude);
    	
        if (flag.equalsIgnoreCase("0xF847")){
        	// Emergencia
        	statusCode = StatusCodes.STATUS_INTRUSION_ON;
      	
        } else if (flag.equalsIgnoreCase("FD13")){
        	// Falla de Energia
        	statusCode = StatusCodes.STATUS_POWER_FAILURE;
        	
        } else if (flag.equalsIgnoreCase("FD15")){
        	// Energia Reconectada
        	statusCode = StatusCodes.STATUS_POWER_RESTORED;
        	
        } 
        /* Alerta de Velocidad MTC - SUTRAN */
    	speedKPH = this.gpsEvent.ajustaVelocidad(device.getDescription(), speedKPH);
    	this.lastModemID 	=	modemID;
        
        String accountID = device.getAccountID();
        String deviceID  = device.getDeviceID();
        String uniqueID  = device.getUniqueID();
        System.out.println("Registro ---> [" + accountID + "/" + deviceID + "/" + uniqueID + "]");
        System.out.println("Evento: [" + new DateTime(fixtime) + "]");
    	        
        this.gpsEvent.setTimestamp(fixtime);
    	this.gpsEvent.setStatusCode(statusCode);
    	this.gpsEvent.setLatitude(latitude);
    	this.gpsEvent.setLongitude(longitude);
    	this.gpsEvent.setSpeedKPH(speedKPH);
    	this.gpsEvent.setHeading(heading);
    	this.gpsEvent.setBatteryLevel(100);
    	this.gpsEvent.setAltitude(altitudeM);
    	
    	/* insert/return */
    	if (this.parseInsertRecord_Common(this.gpsEvent)) {
    		// change this to return any required acknowledgement (ACK) packets back to the device
    		return null;
    	} else {
    		return null;
    	}
    }

    private byte[] parseInsertRecordUpdater(String s)
    {
    	// >id,cellphone<
    	System.out.println("MTC Updater");
    	System.out.println("---------------------------------");
    	String  fld[] = StringTools.parseStringArray(s, ',');
    	if (fld == null) {
    		Print.logWarn("Fields are null");
    		return null;
    	}
    	
    	String modemID = fld[1].trim();
    	long fixtime = (new DateTime(DateTime.getGMTTimeZone())).getTimeSec();//getDayStart() + gpsTOD;
        if ((fixtime - DateTime.MinuteSeconds(15)) > DateTime.getCurrentTimeSec()) {
            fixtime -= DateTime.DaySeconds(1);
        }
    	int		statusCode	= StatusCodes.STATUS_LAST_LOCATION;
    	double latitude   	= 0.0;
    	double longitude 	= 0.0;
    	double speedKPH  	= 0.0;
    	double heading		= 0.0;
    	double altitudeM	= 0.0;
    	//String flag			= fld[2];
    	
    	latitude = StringTools.parseDouble(fld[2], 0.0);
    	longitude = StringTools.parseDouble(fld[3], 0.0);
    	
    	 /* GPS Event */
        this.gpsEvent = this.createGPSEvent(modemID);
        if (this.gpsEvent == null) {
            // errors already displayed
            return null;
        }
    	
        
    	Device device = this.gpsEvent.getDevice();
    	if (device == null) {
    		return null;
    	}    	
    	
    	if (!GeoPoint.isValid(latitude,longitude)) {
            Print.logWarn("Invalid lat/lon: " + latitude + "/" + longitude);
            latitude   = 0.0;
            longitude  = 0.0;
            speedKPH   = 0.0;
            return null;
        }
    	
    	/* Alerta de Velocidad MTC - SUTRAN */
    	speedKPH = this.gpsEvent.ajustaVelocidad(device.getDescription(), speedKPH);
    	//this.lastModemID 	=	modemID;

        String accountID = device.getAccountID();
        String deviceID  = device.getDeviceID();
        String uniqueID  = device.getUniqueID();
        System.out.println("Registro ---> [" + accountID + "/" + deviceID + "/" + uniqueID + "]");
        System.out.println("Evento: [" + new DateTime(fixtime) + "]");
    	        
    	/* Llenamos tabla en Eventos */
        this.gpsEvent.setTimestamp(fixtime);
    	this.gpsEvent.setStatusCode(statusCode);
    	this.gpsEvent.setLatitude(latitude);
    	this.gpsEvent.setLongitude(longitude);
    	this.gpsEvent.setSpeedKPH(speedKPH);
    	this.gpsEvent.setHeading(heading);
    	this.gpsEvent.setBatteryLevel(100);
    	this.gpsEvent.setAltitude(altitudeM);
    	
    	/* insert/return */
    	if (this.parseInsertRecord_Common(this.gpsEvent)) {
    		// change this to return any required acknowledgement (ACK) packets back to the device
    		return null;
    	} else {
    		return null;
    	}
    }

    /* parse and insert data record (common) */
    private boolean parseInsertRecord_Common(GPSEvent gpsEv)
    {
        long   fixtime    = gpsEv.getTimestamp();
        int    statusCode = gpsEv.getStatusCode();
        Device dev        = gpsEv.getDevice(); // guaranteed non-null here

        /* invalid date? */
        if (fixtime <= 0L) {
            Print.logWarn("Invalid date/time");
            fixtime = DateTime.getCurrentTimeSec(); // default to now
            gpsEv.setTimestamp(fixtime);
            return false;
        }
                
        /* valid lat/lon? */
        if (!gpsEv.isValidGeoPoint()) {
            Print.logWarn("Invalid lat/lon: " + gpsEv.getLatitude() + "/" + gpsEv.getLongitude());
            gpsEv.setLatitude(0.0);
            gpsEv.setLongitude(0.0);
            return false;
        }
        GeoPoint geoPoint = gpsEv.getGeoPoint();

        /* minimum speed */
        if (gpsEv.getSpeedKPH() < MINIMUM_SPEED_KPH) {
            gpsEv.setSpeedKPH(0.0);
            gpsEv.setHeading(0.0);
        }

        /* estimate GPS-based odometer */
        double odomKM = 0.0; // set to available odometer from event record
        //if (this.gpsDevice.getLastEventTimestamp() < fixtime) {
        if (odomKM <= 0.0) {
            odomKM = (ESTIMATE_ODOMETER && geoPoint.isValid())? 
                this.gpsDevice.getNextOdometerKM(geoPoint) : 
                this.gpsDevice.getLastOdometerKM();
        } else {
            odomKM = dev.adjustOdometerKM(odomKM);
        }
        //}
        Print.logInfo("Odometer KM: " + odomKM);
        gpsEv.setOdometerKM(odomKM);

        /* simulate Geozone arrival/departure */
        if (SIMEVENT_GEOZONES && geoPoint.isValid()) {
            java.util.List<Device.GeozoneTransition> zone = dev.checkGeozoneTransitions(fixtime, geoPoint);
            if (zone != null) {
                for (Device.GeozoneTransition z : zone) {
                    gpsEv.insertEventData(z.getTimestamp(), z.getStatusCode(), z.getGeozone());
                    Print.logInfo("Geozone    : " + z);
                }
            }
        }

        /* digital input change events */
        if (gpsEv.hasInputMask() && (gpsEv.getInputMask() >= 0L)) {
            long gpioInput = gpsEv.getInputMask();
            if (SIMEVENT_DIGITAL_INPUTS > 0L) {
                // The current input state is compared to the last value stored in the Device record.
                // Changes in the input state will generate a synthesized event.
                long chgMask = (dev.getLastInputState() ^ gpioInput) & SIMEVENT_DIGITAL_INPUTS;
                if (chgMask != 0L) {
                    // an input state has changed
                    for (int b = 0; b <= 15; b++) {
                        long m = 1L << b;
                        if ((chgMask & m) != 0L) {
                            // this bit changed
                            int  inpCode = ((gpioInput & m) != 0L)? InputStatusCodes_ON[b] : InputStatusCodes_OFF[b];
                            long inpTime = fixtime;
                            gpsEv.insertEventData(inpTime, inpCode);
                            Print.logInfo("GPIO : " + StatusCodes.GetDescription(inpCode,null));
                        }
                    }
                }
            }
            dev.setLastInputState(gpioInput & 0xFFFFL); // FLD_lastInputState
        }

        /* create/insert standard event */
        gpsEv.insertEventData(fixtime, statusCode);

        /* save Device changes */
        gpsEv.updateDevice();

        /* return success */
        return true;

    }

    /**
    *** Initialize runtime configuration
    **/
    public static void configInit() 
    {
        DCServerConfig dcsc     = Main.getServerConfig(null);
        if (dcsc == null) {
            Print.logWarn("DCServer not found: " + Main.getServerName());
            return;
        }

        /* custom */
        DATA_FORMAT_OPTION      = dcsc.getIntProperty(Main.ARG_FORMAT, DATA_FORMAT_OPTION);

        /* common */
        MINIMUM_SPEED_KPH       = dcsc.getMinimumSpeedKPH(MINIMUM_SPEED_KPH);
        ESTIMATE_ODOMETER       = dcsc.getEstimateOdometer(ESTIMATE_ODOMETER);
        SIMEVENT_GEOZONES       = dcsc.getSimulateGeozones(SIMEVENT_GEOZONES);
        SIMEVENT_DIGITAL_INPUTS = dcsc.getSimulateDigitalInputs(SIMEVENT_DIGITAL_INPUTS) & 0xFFFFL;

    }

    // ------------------------------------------------------------------------
    // ------------------------------------------------------------------------
    // ------------------------------------------------------------------------
    // Once you have modified this example 'template' server to parse your particular
    // Device packets, you can also use this source module to load GPS data packets
    // which have been saved in a file.  To run this module to load your save GPS data
    // packets, start this command as follows:
    //   java -cp <classpath> org.opengts.servers.template.TrackClientPacketHandler {options}
    // Where your options are one or more of 
    //   -insert=[true|false]    Insert parse records into EventData
    //   -format=[1|2]           Data format
    //   -debug                  Parse internal sample data
    //   -parseFile=<file>       Parse data from specified file

    private static int _usage()
    {
        String cn = StringTools.className(TrackClientPacketHandler.class);
        Print.sysPrintln("Test/Load Device Communication Server");
        Print.sysPrintln("Usage:");
        Print.sysPrintln("  $JAVA_HOME/bin/java -classpath <classpath> %s {options}", cn);
        Print.sysPrintln("Options:");
        Print.sysPrintln("  -insert=[true|false]    Insert parsed records into EventData");
        Print.sysPrintln("  -format=[1|2]           Data format");
        Print.sysPrintln("  -debug                  Parse internal sample/debug data (if any)");
        Print.sysPrintln("  -parseFile=<file>       Parse data from specified file");
        return 1;
    }

    /**
    *** Main entry point used for debug purposes only
    **/
    public static int _main(boolean fromMain)
    {

        /* default options */
        INSERT_EVENT = RTConfig.getBoolean(Main.ARG_INSERT, DFT_INSERT_EVENT);
        if (!INSERT_EVENT) {
            Print.sysPrintln("Warning: Data will NOT be inserted into the database");
        }

        /* create client packet handler */
        TrackClientPacketHandler tcph = new TrackClientPacketHandler();

        /* DEBUG sample data */
        if (RTConfig.getBoolean(Main.ARG_DEBUG,false)) {
            String data[] = null;
            switch (DATA_FORMAT_OPTION) {
                case  1: data = new String[] {
                    "123456789012345,2006/09/05,07:47:26,35.3640,-142.2958,27.0,224.8",
                }; break;
                case  2: data = new String[] {
                    "account/device/$GPRMC,025423.494,A,3709.0642,N,14207.8315,W,12.09,108.52,200505,,*2E",
                    "/device/$GPRMC,025423.494,A,3709.0642,N,14207.8315,W,12.09,108.52,200505,,*2E",
                }; break;
                case  3: data = new String[] {
                    "2,123,1234567890,0,20101223,110819,1,2.1,39.1234,-142.1234,33,227,1800",
                }; break;
                case  9: data = new String[] {
                    "mid=123456789012345 lat=39.12345 lon=-142.12345 kph=123.0"
                }; break;
                default:
                    Print.sysPrintln("Unrecognized Data Format: %d", DATA_FORMAT_OPTION);
                    return _usage();
            }
            for (int i = 0; i < data.length; i++) {
                tcph.getHandlePacket(data[i].getBytes());
            }
            return 0;
        }

        /* 'parseFile' specified? */
        if (RTConfig.hasProperty(Main.ARG_PARSEFILE)) {

            /* get input file */
            File parseFile = RTConfig.getFile(Main.ARG_PARSEFILE,null);
            if ((parseFile == null) || !parseFile.isFile()) {
                Print.sysPrintln("Data source file not specified, or does not exist.");
                return _usage();
            }

            /* open file */
            FileInputStream fis = null;
            try {
                fis = new FileInputStream(parseFile);
            } catch (IOException ioe) {
                Print.logException("Error openning input file: " + parseFile, ioe);
                return 2;
            }

            /* loop through file */
            try {
                // records are assumed to be terminated by CR/NL 
                for (;;) {
                    String data = FileTools.readLine(fis);
                    if (!StringTools.isBlank(data)) {
                        tcph.getHandlePacket(data.getBytes());
                    }
                }
            } catch (EOFException eof) {
                Print.sysPrintln("");
                Print.sysPrintln("***** End-Of-File *****");
            } catch (IOException ioe) {
                Print.logException("Error reaading input file: " + parseFile, ioe);
            } finally {
                try { fis.close(); } catch (Throwable th) {/* ignore */}
            }

            /* done */
            return 0;

        }

        /* no options? */
        return _usage();

    }

    /**
    *** Main entry point used for debug purposes only
    **/
    public static void main(String argv[])
    {
        DBConfig.cmdLineInit(argv,false);
        TrackClientPacketHandler.configInit();
        System.exit(TrackClientPacketHandler._main(false));
    }
    
}
