package org.opengts.servers.dct;

import java.io.*;
import java.net.*;
import org.opengts.util.*;
import org.opengts.db.*;
import org.opengts.db.tables.*;
import org.opengts.servers.*;

/**
*** <code>TrackClientPacketHandler</code> - This module contains the general
*** "business logic" for parsing incoming data packets from the remote tracking
*** Device.
**/

public class TrackClientPacketHandler
    extends AbstractClientPacketHandler
{
    public  static 		int	DATA_FORMAT_OPTION				= 1;

    //    vehicle value.
    public  static		boolean ESTIMATE_ODOMETER          = true;
    
    /* simulate geozone arrival/departure */
    // (enable to insert simulated Geozone arrival/departure EventData records)
    public  static       boolean SIMEVENT_GEOZONES          = true;
    
    /* simulate digital input changes */
    public  static       long    SIMEVENT_DIGITAL_INPUTS    = 0x0000L; // 0xFFFFL;

    /* flag indicating whether data should be inserted into the DB */
    // should be set to 'true' for production.
    private static       boolean DFT_INSERT_EVENT           = true;
    private static       boolean INSERT_EVENT               = DFT_INSERT_EVENT;

    /* update Device record */
    // (enable to update Device record with current IP address and last connect time)
  //private static       boolean UPDATE_DEVICE              = false;
    
    /* minimum acceptable speed value */
    // Speeds below this value should be considered 'stopped'
    public  static       double  MINIMUM_SPEED_KPH          = 0.0;

    /* Knot/Kilometer conversions */
    public static final double  KILOMETERS_PER_KNOT     = 1.85200000;
    public static final double  KNOTS_PER_KILOMETER     = 1.0 / KILOMETERS_PER_KNOT;

    // ------------------------------------------------------------------------
    
    private 			String 		tipoData				=	"";

    /* GTS status codes for Input-On events */
    private static final int InputStatusCodes_ON[] = new int[] {
        StatusCodes.STATUS_INPUT_ON_00,
        StatusCodes.STATUS_INPUT_ON_01,
        StatusCodes.STATUS_INPUT_ON_02,
        StatusCodes.STATUS_INPUT_ON_03,
        StatusCodes.STATUS_INPUT_ON_04,
        StatusCodes.STATUS_INPUT_ON_05,
        StatusCodes.STATUS_INPUT_ON_06,
        StatusCodes.STATUS_INPUT_ON_07,
        StatusCodes.STATUS_INPUT_ON_08,
        StatusCodes.STATUS_INPUT_ON_09,
        StatusCodes.STATUS_INPUT_ON_10,
        StatusCodes.STATUS_INPUT_ON_11,
        StatusCodes.STATUS_INPUT_ON_12,
        StatusCodes.STATUS_INPUT_ON_13,
        StatusCodes.STATUS_INPUT_ON_14,
        StatusCodes.STATUS_INPUT_ON_15
    };

    /* GTS status codes for Input-Off events */
    private static final int InputStatusCodes_OFF[] = new int[] {
        StatusCodes.STATUS_INPUT_OFF_00,
        StatusCodes.STATUS_INPUT_OFF_01,
        StatusCodes.STATUS_INPUT_OFF_02,
        StatusCodes.STATUS_INPUT_OFF_03,
        StatusCodes.STATUS_INPUT_OFF_04,
        StatusCodes.STATUS_INPUT_OFF_05,
        StatusCodes.STATUS_INPUT_OFF_06,
        StatusCodes.STATUS_INPUT_OFF_07,
        StatusCodes.STATUS_INPUT_OFF_08,
        StatusCodes.STATUS_INPUT_OFF_09,
        StatusCodes.STATUS_INPUT_OFF_10,
        StatusCodes.STATUS_INPUT_OFF_11,
        StatusCodes.STATUS_INPUT_OFF_12,
        StatusCodes.STATUS_INPUT_OFF_13,
        StatusCodes.STATUS_INPUT_OFF_14,
        StatusCodes.STATUS_INPUT_OFF_15
    };

    // ------------------------------------------------------------------------
    // ------------------------------------------------------------------------

    /* TCP session ID */
    private String          sessionID                   = null;

    /* common GPSEvent instance */
    private GPSEvent        gpsEvent                    = null;

    /* Device record */
    private Device          gpsDevice                   = null;
    private String          lastModemID                 = null;

    /* Session 'terminate' indicator */
    // This value should be set to 'true' when this server has determined that the
    // session should be terminated.  For instance, if this server finishes communication
    // with the Device or if parser finds a fatal error in the incoming data stream 
    // (ie. invalid Account/Device, or unrecognizable data).
    private boolean         terminate                   = false;

    /* session IP address */
    // These values will be set for you by the incoming session to indicate the 
    // originating IP address.
    private String          ipAddress                   = null;
    private int             clientPort                  = 0;

    /* packet handler constructor */
    public TrackClientPacketHandler() 
    {
        super();
        //Print.logStackTrace("new TrackClientPacketHandler ...");
    }

    // ------------------------------------------------------------------------

    /* callback when session is starting */
    // this method is called at the beginning of a communication session
    public void sessionStarted(InetAddress inetAddr, boolean isTCP, boolean isText)
    {
        super.sessionStarted(inetAddr, isTCP, isText);
        super.clearTerminateSession();

        /* init */
        this.ipAddress        = (inetAddr != null)? inetAddr.getHostAddress() : null;
        this.clientPort       = this.getSessionInfo().getRemotePort();
        if (this.isDuplex()) {
        	this.tipoData="TCP";
            //Print.logInfo("Inicio TCP: " + this.ipAddress + " [" + new DateTime() + "]");
        	//	Print.logInfo("========================================================");
        } else {
        	this.tipoData="UDP";
        	//Print.logInfo("Inicio UDP: " + this.ipAddress + " [" + new DateTime() + "]");
        	//Print.logInfo("========================================================");
        }

    }

    /* callback when session is terminating */
    // this method is called at the end of a communication session
    public void sessionTerminated(Throwable err, long readCount, long writeCount)
    {
        super.sessionTerminated(err, readCount, writeCount);
    }

    /* callback to return the TCP session id */
    public String getSessionID()
    {
        if (!StringTools.isBlank(this.sessionID)) {
            return this.sessionID;
        } else
        if (this.gpsDevice != null) {
            return CreateTcpSessionID(this.gpsDevice);
        } else
        if (this.gpsEvent != null) {
            return CreateTcpSessionID(this.gpsEvent.getDevice());
        } else {
            return null;
        }
    }

    // ------------------------------------------------------------------------

    public static final boolean USE_STANDARD_TCP_SESSION_ID = true;

    /* get TCP session ID */
    public static String GetTcpSessionID(Device dev)
    {
        if (USE_STANDARD_TCP_SESSION_ID || (dev == null)) {
            return DCServerFactory.getTcpSessionID(dev);
        } else {
            // Custom example for extracting the session ID from the Device 
            // UniqueID (instead of the DeviceID)
            String uid = dev.getUniqueID();
            int    u   = uid.lastIndexOf("_");
            if (u < 0) {
                // no "_" character, fallback to standard TCP sessionID
                return DCServerFactory.getTcpSessionID(dev);
            } else {
                String sessID = dev.getAccountID() + "/" + uid.substring(0,u+1);
                return sessID;
            }
        }
    }

    /* create TCP session ID */
    public static String CreateTcpSessionID(Device dev)
    {
        if (USE_STANDARD_TCP_SESSION_ID || (dev == null)) {
            return DCServerFactory.createTcpSessionID(dev);
        } else {
            String uid = dev.getUniqueID();
            int    u   = uid.lastIndexOf("_");
            if (u < 0) {
                // no "_" character, fallback to standard TCP sessionID
                return DCServerFactory.createTcpSessionID(dev);
            } else {
                String sessID = dev.getAccountID() + "/" + uid.substring(0,u+1);
                dev.setLastTcpSessionID(sessID);
                return sessID;
            }
        }
    }

     /**
     *** Create GPSEvent from the ModemID
     *** @return The created GPSEvent instance, or null if the ModemID is invalid
     **/
     private GPSEvent createGPSEvent(String modemID)
     {
         DCServerConfig dcserver = Main.getServerConfig(null);

         /* create/load device */
         if (this.gpsDevice != null) {
             if (!modemID.equals("*") && 
                 ((this.lastModemID == null) || !this.lastModemID.equals(modemID))) { // fix [B10]
                 Print.logError("New MobileID does not match previously loaded Device");
                 //return this.gpsEvent;
                 return null;
             }
             this.gpsEvent  = new GPSEvent(dcserver, this.ipAddress, this.clientPort, this.gpsDevice);
         } else {
             if (StringTools.isBlank(modemID) || modemID.equals("*")) {
                 Print.logWarn("ModemID not specified!");
                 return null;
             }
             this.gpsEvent  = new GPSEvent(dcserver, this.ipAddress, this.clientPort, modemID);
             this.gpsDevice = this.gpsEvent.getDevice(); // may still be null
         }

         /* no Device? */
         if (this.gpsDevice == null) {
             // errors already displayed
             return null;
         }

         /* create session ID */
         this.sessionID = CreateTcpSessionID(this.gpsDevice);

         /* return GPSEvent */
         return this.gpsEvent; // non-null

     }

    /* based on the supplied packet data, return the remaining bytes to read in the packet */
    public int getActualPacketLength(byte packet[], int packetLen)
    {
        if (Constants.ASCII_PACKETS) {
            // (this actually won't be called if 'Constants.ASCII_PACKETS' is true).
            // ASCII packets - look for line terminator [see Constants.ASCII_LINE_TERMINATOR)]
            return ServerSocketThread.PACKET_LEN_LINE_TERMINATOR;  // read until line termination character
            //return ServerSocketThread.PACKET_LEN_END_OF_STREAM;  // read until end of stream, or maxlen
        } else {
            // BINARY packet - need to analyze 'packet[]' and determine actual packet length
            //return ServerSocketThread.PACKET_LEN_LINE_TERMINATOR; // <-- change this for binary packets
        	return ServerSocketThread.PACKET_LEN_END_OF_STREAM; // <-- change this for binary packets
        }
        
    }

    /* set session terminate after next packet handling */
    private void setTerminate()
    {
        this.terminate = true;
    }
    
    /* indicate that the session should terminate */
    // This method is called after each return from "getHandlePacket" to check to see
    // the current session should be closed.
    public boolean getTerminateSession()
    {
        return this.terminate;
    }

    // ------------------------------------------------------------------------
    // ------------------------------------------------------------------------

    /* return the initial packet sent to the Device after session is open */
    public byte[] getInitialPacket() 
        throws Exception
    {
        // At this point a connection from the client to the server has just been
        // initiated, and we have not yet received any data from the client.
        // If the client is expecting to receive an initial packet from the server at
        // the time that the client connects, then this is where the server can return
        // a byte array that will be transmitted to the client Device.
        return null;
        // Note: any returned response for "getInitialPacket()" is ignored for simplex/udp connections.
        // Returned UDP packets may be sent from "getHandlePacket" or "getFinalPacket".
    }

    /* workhorse of the packet handler */
    public byte[] getHandlePacket(byte pktBytes[]) 
    {
        if ((pktBytes != null) && (pktBytes.length > 0)) {
            
        	String s = StringTools.toStringValue(pktBytes).trim(); // remove leading/trailing spaces
            System.out.println("");
            
            byte rtn[] = null;
            
            if (s.startsWith(">REV")){
            	System.out.println(tipoData +" : " + s);
            	rtn = this.insertREV_data(s);
            } else if (s.startsWith(">RPV")){
        		//System.out.println("RPV: " + trama); // debug message
        		rtn = this.parseNull(s);
            } else {
            	return null;
            }
            
            return rtn; // no return packets are expected

        } else {

            /* no packet date received */
            Print.logInfo("...");
            return null; // no return packets are expected

        }
    }
    
    private byte[] parseNull(String s)
    {
                /* pre-validate */
        if (StringTools.isBlank(s)) {
            Print.logError("String is null/blank");
            return null;
        } else
        if (s.length() < 5) {
            Print.logError("String is invalid length");
            return null;
        } else
        if (!s.startsWith(">")) {
            Print.logError("String does not start with '>'");
            return null;
        }

        /* ends with "<"? */
        int se = s.endsWith("<")? (s.length() - 1) : s.length();
        s = s.substring(1,se);

        /* split */
        String T[] = StringTools.split(s,';');

        /* handle ">RPV" records only */
        if (!T[0].startsWith("REV")) {
            Print.logWarn("Only 'REV' record types please!");
            this.setTerminateSession();
            return null;
        }

        /* RPV record */
        if (T[0].length() < 40) {
            Print.logError("Invalid 'REV' data length");
            return null;
        }
        
        return null;

    }  
    
 // ------------------------------------------------------------------------
    private byte[] insertREV_data(String s)
    {
        //	>RPV21305+3958635-1424085300000012;ID=0011<
        //	>REV001147209057-1198767-0770648300000011;ID=356612020487833<
        // Example:
        // -------
        //	>RPV09057-1198767-0770648300000011;ID=356612020487833<
        //   R          = [ 0, 1] Response Query
        //   EV         = [ 1, 3] Position/Velocity
        //	 00			= [ 3, 5] Event ID
        // 	 1147		= [ 5, 9] Numero de semanas desde el 00:00AM 6 Enero 1980
        //	 2			= [ 9,10] Numero de dias en esa semana 0= domingo
        //   09057      = [10,15] GPS time-of-day
        //   -1198767   = [15,23] Latitude
        //   -07706483  = [23,32] Longitude
        //   000        = [32,35] Speed (mph)
        //   000        = [35,38] Heading (degrees)
        //   1          = [38,39] GPS source [0=2D-GPS, 1=3D-GPS, 2=2D-DGPS, 3=3D-DGPS, 6=DR, 8=Degraded-DR, 9=Unknown]
        //   1          = [39,40] Age [0=n/a, 1=old, 2=fresh]
        //   ;ID=1234   = VehicleID
        //   ;*7F       = Checksum
        //   <          = End of message

        /* pre-validate */
        if (StringTools.isBlank(s)) {
            Print.logError("String is null/blank");
            return null;
        } else
        if (s.length() < 5) {
            Print.logError("String is invalid length");
            return null;
        } else
        if (!s.startsWith(">")) {
            Print.logError("String does not start with '>'");
            return null;
        }

        /* ends with "<"? */
        int se = s.endsWith("<")? (s.length() - 1) : s.length();
        s = s.substring(1,se);

        /* split */
        String T[] = StringTools.split(s,';');

        /* handle ">RPV" records only */
        if (!T[0].startsWith("REV")) {
            Print.logWarn("Solo registros REV por favor!");
            return null;
        }

        /* RPV record */
        if (T[0].length() < 40) {
            Print.logError("Invalid 'REV' data length");
            return null;
        }

        /* mobile id */
        String modemID = null;
        for (int i = 1; i < T.length; i++) {
            if (T[i].startsWith("ID=")) {
                modemID = T[i].substring(3);
                modemID=modemID.toLowerCase();
                break;
            }
        }
        //imeiID = modemID;
       
        int	   eventID	= StringTools.parseInt(T[0].substring(3, 5), 0);
        int    numSem	= StringTools.parseInt(T[0].substring(5, 9), 0);
        int    numDia	= StringTools.parseInt(T[0].substring(9, 10), 0);
        int    statusCode = StatusCodes.STATUS_LOCATION;
        int    gpsTOD     = StringTools.parseInt(T[0].substring(10,15), 0);
        double latitude   = (double)StringTools.parseLong(T[0].substring(15,23),0L) / 100000.0;
        double longitude  = (double)StringTools.parseLong(T[0].substring(23,32),0L) / 100000.0;
        double speedKPH   = StringTools.parseDouble(T[0].substring(32,35), 0.0) * GeoPoint.KILOMETERS_PER_MILE;
        double heading	  = StringTools.parseDouble(T[0].substring(35,38), 0.0);
        String srcStr     = T[0].substring(38,39);
        String ageStr     = T[0].substring(39,40);
        double altitudeM  = 0.0;
        long   gpioInput  = 0L;

        /* Fix time */
        long prueba0  = (new DateTime(DateTime.getGMTTimeZone())).getDayStart();
        long prueba  = (new DateTime(DateTime.getGMTTimeZone())).getTimeSec();
        long fixtime = (new DateTime(DateTime.getGMTTimeZone())).getTimeSec();//getDayStart() + gpsTOD;
        if ((fixtime - DateTime.MinuteSeconds(15)) > DateTime.getCurrentTimeSec()) {
            fixtime -= DateTime.DaySeconds(1);
        }

        /* lat/lon valid? */
        boolean validGPS = true;
        if (!GeoPoint.isValid(latitude,longitude)) {
            Print.logWarn("Invalid lat/lon: " + latitude + "/" + longitude);
            validGPS   = false;
            latitude   = 0.0;
            longitude  = 0.0;
            speedKPH   = 0.0;
            heading	   = 0.0;
        }

        /* adjustments to received values */
        if (speedKPH < MINIMUM_SPEED_KPH) {
            speedKPH   = 0.0;
            heading = 0.0;
        } else
        if (heading < 0.0) {
            heading = 0.0;
        }

        /* debug */
        //System.out.println("MobileID  : " + mobileID);
        //System.out.println("Timestamp : " + new DateTime(fixtime));
        

        /* mobile-id */
        if (StringTools.isBlank(modemID)) {
            Print.logError("Missing MobileID");
            return null;
        }

        /* GPS Event */
        this.gpsEvent = this.createGPSEvent(modemID);
        if (this.gpsEvent == null) {
            // errors already displayed
            return null;
        }
    	
    	Device device = this.gpsEvent.getDevice();
    	if (device == null) {
    		return null;
    	}
    	String accountID = device.getAccountID();
        String deviceID  = device.getDeviceID();
        String uniqueID  = device.getUniqueID();
        System.out.println("Registro ---> [" + accountID + "/" + deviceID + "/" + uniqueID + "]");
        System.out.println("Evento: " + fixtime + " [" + new DateTime(fixtime) + "]");

        /* reject invalid GPS fixes? */
        if (!validGPS && (statusCode == StatusCodes.STATUS_LOCATION)) {
            // ignore invalid GPS fixes that have a simple 'STATUS_LOCATION' status code
            Print.logWarn("Ignoring event with invalid latitude/longitude");
            return null;
        }
        
        System.out.println("EventID: " + eventID);
        
        if (eventID==1){
        	if (speedKPH == 0){
        		statusCode	=	StatusCodes.STATUS_MOTION_STOP;	
        	} else if (speedKPH >= 6){
        		statusCode	=	StatusCodes.STATUS_MOTION_START;
        	} else if (speedKPH >= 20){
        		statusCode	=	StatusCodes.STATUS_MOTION_IN_MOTION;
        	} else if (speedKPH >= 90){
        		statusCode	=	StatusCodes.STATUS_MOTION_EXCESS_SPEED;
        	} else {
        		statusCode	=	StatusCodes.STATUS_MOTION_HEADING;
        	}
        } else if (eventID==2){
        	statusCode	=	StatusCodes.STATUS_IGNITION_ON;
        } else if (eventID==3){
        	statusCode	=	StatusCodes.STATUS_IGNITION_OFF;
        } else if (eventID==4){
        	statusCode	=	StatusCodes.STATUS_PANIC_ON;
        } else if (eventID==5){
        	statusCode	=	StatusCodes.STATUS_INPUT_ON_02;
        } else if (eventID==6){
        	statusCode	=	StatusCodes.STATUS_INPUT_ON_03;
        } else if (eventID==8){
        	statusCode	=	StatusCodes.STATUS_POWER_FAILURE;
        } else if (eventID==9){
        	statusCode	=	StatusCodes.STATUS_POWER_RESTORED;
        } else if (eventID==10){
        	statusCode	=	StatusCodes.STATUS_MOTION_EXCESS_SPEED;
        } else if (eventID==11){
        	statusCode	=	StatusCodes.STATUS_MOTION_IDLE;
        } else if (eventID==13){
        	statusCode	=	StatusCodes.STATUS_GPS_ANTENNA_OPEN;
        } else if (eventID==14){
        	statusCode	=	StatusCodes.STATUS_GPS_ANTENNA_SHORT;
        } else if (eventID==15){
        	statusCode	=	StatusCodes.STATUS_LOW_BATTERY;
        } else if (eventID==17){
        	statusCode	=	StatusCodes.STATUS_MOTION_ACCELERATION;
        } else if (eventID==18){
        	statusCode	=	StatusCodes.STATUS_IMPACT;
        } else if (eventID==46){
        	statusCode	=	StatusCodes.STATUS_LOW_BATTERY;
        } else if (eventID==53){
        	statusCode	=	StatusCodes.STATUS_GPS_JAMMING;
        } else if (eventID==56){
        	statusCode	=	StatusCodes.STATUS_MOTION_EXCESS_SPEED;
        } else {
        	statusCode	=	StatusCodes.STATUS_LOCATION;
        }
        
        /* Alerta de Velocidad MTC - SUTRAN */
    	speedKPH = this.gpsEvent.ajustaVelocidad(device.getDescription(), speedKPH);
    	this.lastModemID 	=	modemID;
    	
    	this.gpsEvent.setTimestamp(fixtime);
    	this.gpsEvent.setStatusCode(statusCode);
    	this.gpsEvent.setLatitude(latitude);
    	this.gpsEvent.setLongitude(longitude);
    	this.gpsEvent.setSpeedKPH(speedKPH);
    	this.gpsEvent.setHeading(heading);
    	this.gpsEvent.setBatteryLevel(100);
    	this.gpsEvent.setAltitude(altitudeM);
    	
    	
    	/* insert/return */
    	if (this.parseInsertRecord_Common(this.gpsEvent)) {
    		// change this to return any required acknowledgement (ACK) packets back to the device
    		return modemID.getBytes();
    		//return null;
    	} else {
    		return null;
    	}

    }

    /* final packet sent to Device before session is closed */
    public byte[] getFinalPacket(boolean hasError) 
        throws Exception
    {
        return null;
    }
    
    /* parse and insert data record (common) */
    private boolean parseInsertRecord_Common(GPSEvent gpsEv)
    {
        long   fixtime    = gpsEv.getTimestamp();
        int    statusCode = gpsEv.getStatusCode();
        Device dev        = gpsEv.getDevice(); // guaranteed non-null here

        /* invalid date? */
        if (fixtime <= 0L) {
            Print.logWarn("Invalid date/time");
            fixtime = DateTime.getCurrentTimeSec(); // default to now
            gpsEv.setTimestamp(fixtime);
        }
                
        /* valid lat/lon? */
        if (!gpsEv.isValidGeoPoint()) {
            Print.logWarn("Invalid lat/lon: " + gpsEv.getLatitude() + "/" + gpsEv.getLongitude());
            gpsEv.setLatitude(0.0);
            gpsEv.setLongitude(0.0);
        }
        GeoPoint geoPoint = gpsEv.getGeoPoint();

        /* minimum speed */
        if (gpsEv.getSpeedKPH() < MINIMUM_SPEED_KPH) {
            gpsEv.setSpeedKPH(0.0);
            gpsEv.setHeading(0.0);
        }

        /* estimate GPS-based odometer */
        double odomKM = 0.0; // set to available odometer from event record
        //if (this.gpsDevice.getLastEventTimestamp() < fixtime) {
        if (odomKM <= 0.0) {
            odomKM = (ESTIMATE_ODOMETER && geoPoint.isValid())? 
                this.gpsDevice.getNextOdometerKM(geoPoint) : 
                this.gpsDevice.getLastOdometerKM();
        } else {
            odomKM = dev.adjustOdometerKM(odomKM);
        }
        //}
        Print.logInfo("Odometer KM: " + odomKM);
        gpsEv.setOdometerKM(odomKM);

        /* simulate Geozone arrival/departure */
        if (SIMEVENT_GEOZONES && geoPoint.isValid()) {
            java.util.List<Device.GeozoneTransition> zone = dev.checkGeozoneTransitions(fixtime, geoPoint);
            if (zone != null) {
                for (Device.GeozoneTransition z : zone) {
                    gpsEv.insertEventData(z.getTimestamp(), z.getStatusCode(), z.getGeozone());
                    Print.logInfo("Geozone    : " + z);
                }
            }
        }

        /* digital input change events */
        if (gpsEv.hasInputMask() && (gpsEv.getInputMask() >= 0L)) {
            long gpioInput = gpsEv.getInputMask();
            if (SIMEVENT_DIGITAL_INPUTS > 0L) {
                // The current input state is compared to the last value stored in the Device record.
                // Changes in the input state will generate a synthesized event.
                long chgMask = (dev.getLastInputState() ^ gpioInput) & SIMEVENT_DIGITAL_INPUTS;
                if (chgMask != 0L) {
                    // an input state has changed
                    for (int b = 0; b <= 15; b++) {
                        long m = 1L << b;
                        if ((chgMask & m) != 0L) {
                            // this bit changed
                            int  inpCode = ((gpioInput & m) != 0L)? InputStatusCodes_ON[b] : InputStatusCodes_OFF[b];
                            long inpTime = fixtime;
                            gpsEv.insertEventData(inpTime, inpCode);
                            Print.logInfo("GPIO : " + StatusCodes.GetDescription(inpCode,null));
                        }
                    }
                }
            }
            dev.setLastInputState(gpioInput & 0xFFFFL); // FLD_lastInputState
        }

        /* create/insert standard event */
        gpsEv.insertEventData(fixtime, statusCode);

        /* save Device changes */
        gpsEv.updateDevice();

        /* return success */
        return true;

    }

    /**
    *** Initialize runtime configuration
    **/
    public static void configInit() 
    {
        DCServerConfig dcsc     = Main.getServerConfig(null);
        if (dcsc == null) {
            Print.logWarn("DCServer not found: " + Main.getServerName());
            return;
        }

        /* custom */
        DATA_FORMAT_OPTION      = dcsc.getIntProperty(Main.ARG_FORMAT, DATA_FORMAT_OPTION);

        /* common */
        MINIMUM_SPEED_KPH       = dcsc.getMinimumSpeedKPH(MINIMUM_SPEED_KPH);
        ESTIMATE_ODOMETER       = dcsc.getEstimateOdometer(ESTIMATE_ODOMETER);
        SIMEVENT_GEOZONES       = dcsc.getSimulateGeozones(SIMEVENT_GEOZONES);
        SIMEVENT_DIGITAL_INPUTS = dcsc.getSimulateDigitalInputs(SIMEVENT_DIGITAL_INPUTS) & 0xFFFFL;

    }

    // ------------------------------------------------------------------------
    // ------------------------------------------------------------------------
    // ------------------------------------------------------------------------
    // Once you have modified this example 'template' server to parse your particular
    // Device packets, you can also use this source module to load GPS data packets
    // which have been saved in a file.  To run this module to load your save GPS data
    // packets, start this command as follows:
    //   java -cp <classpath> org.opengts.servers.template.TrackClientPacketHandler {options}
    // Where your options are one or more of 
    //   -insert=[true|false]    Insert parse records into EventData
    //   -format=[1|2]           Data format
    //   -debug                  Parse internal sample data
    //   -parseFile=<file>       Parse data from specified file

    private static int _usage()
    {
        String cn = StringTools.className(TrackClientPacketHandler.class);
        Print.sysPrintln("Test/Load Device Communication Server");
        Print.sysPrintln("Usage:");
        Print.sysPrintln("  $JAVA_HOME/bin/java -classpath <classpath> %s {options}", cn);
        Print.sysPrintln("Options:");
        Print.sysPrintln("  -insert=[true|false]    Insert parsed records into EventData");
        Print.sysPrintln("  -format=[1|2]           Data format");
        Print.sysPrintln("  -debug                  Parse internal sample/debug data (if any)");
        Print.sysPrintln("  -parseFile=<file>       Parse data from specified file");
        return 1;
    }

    /**
    *** Main entry point used for debug purposes only
    **/
    public static int _main(boolean fromMain)
    {

        /* default options */
        INSERT_EVENT = RTConfig.getBoolean(Main.ARG_INSERT, DFT_INSERT_EVENT);
        if (!INSERT_EVENT) {
            Print.sysPrintln("Warning: Data will NOT be inserted into the database");
        }

        /* create client packet handler */
        TrackClientPacketHandler tcph = new TrackClientPacketHandler();

        /* DEBUG sample data */
        if (RTConfig.getBoolean(Main.ARG_DEBUG,false)) {
            String data[] = null;
            switch (DATA_FORMAT_OPTION) {
                case  1: data = new String[] {
                    "123456789012345,2006/09/05,07:47:26,35.3640,-142.2958,27.0,224.8",
                }; break;
                case  2: data = new String[] {
                    "account/device/$GPRMC,025423.494,A,3709.0642,N,14207.8315,W,12.09,108.52,200505,,*2E",
                    "/device/$GPRMC,025423.494,A,3709.0642,N,14207.8315,W,12.09,108.52,200505,,*2E",
                }; break;
                case  3: data = new String[] {
                    "2,123,1234567890,0,20101223,110819,1,2.1,39.1234,-142.1234,33,227,1800",
                }; break;
                case  9: data = new String[] {
                    "mid=123456789012345 lat=39.12345 lon=-142.12345 kph=123.0"
                }; break;
                default:
                    Print.sysPrintln("Unrecognized Data Format: %d", DATA_FORMAT_OPTION);
                    return _usage();
            }
            for (int i = 0; i < data.length; i++) {
                tcph.getHandlePacket(data[i].getBytes());
            }
            return 0;
        }

        /* 'parseFile' specified? */
        if (RTConfig.hasProperty(Main.ARG_PARSEFILE)) {

            /* get input file */
            File parseFile = RTConfig.getFile(Main.ARG_PARSEFILE,null);
            if ((parseFile == null) || !parseFile.isFile()) {
                Print.sysPrintln("Data source file not specified, or does not exist.");
                return _usage();
            }

            /* open file */
            FileInputStream fis = null;
            try {
                fis = new FileInputStream(parseFile);
            } catch (IOException ioe) {
                Print.logException("Error openning input file: " + parseFile, ioe);
                return 2;
            }

            /* loop through file */
            try {
                // records are assumed to be terminated by CR/NL 
                for (;;) {
                    String data = FileTools.readLine(fis);
                    if (!StringTools.isBlank(data)) {
                        tcph.getHandlePacket(data.getBytes());
                    }
                }
            } catch (EOFException eof) {
                Print.sysPrintln("");
                Print.sysPrintln("***** End-Of-File *****");
            } catch (IOException ioe) {
                Print.logException("Error reaading input file: " + parseFile, ioe);
            } finally {
                try { fis.close(); } catch (Throwable th) {/* ignore */}
            }

            /* done */
            return 0;

        }

        /* no options? */
        return _usage();

    }

    /**
    *** Main entry point used for debug purposes only
    **/
    public static void main(String argv[])
    {
        DBConfig.cmdLineInit(argv,false);
        TrackClientPacketHandler.configInit();
        System.exit(TrackClientPacketHandler._main(false));
    }
    
}
