// ----------------------------------------------------------------------------
// Copyright 2007-2013, GeoTelematic Solutions, Inc.
// All rights reserved
// ----------------------------------------------------------------------------
//
// This source module is PROPRIETARY and CONFIDENTIAL.
// NOT INTENDED FOR PUBLIC RELEASE.
// 
// Use of this software is subject to the terms and conditions outlined in
// the 'Commercial' license provided with this software.  If you did not obtain
// a copy of the license with this software please request a copy from the
// Software Provider.
// 
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// ----------------------------------------------------------------------------
// Change History:
//  2009/07/01  Martin D. Flynn
//     -Initial release
//  2011/04/01  Martin D. Flynn
//     -Added map support
//  2012/05/27  Martin D. Flynn
//     -Fixed map timestamp issue
//     -Added engine status code support
// ----------------------------------------------------------------------------
package org.opengts.extra.war.report.field;

import java.io.*;
import java.util.*;

import javax.servlet.*;
import javax.servlet.http.*;

import org.opengts.util.*;
import org.opengts.dbtools.*;
import org.opengts.db.*;
import org.opengts.db.tables.*;

import org.opengts.war.tools.*;
import org.opengts.war.report.*;
import org.opengts.war.report.field.*;

public class DailySummaryReport
    extends ReportData
    implements DBRecordHandler<EventData>
{

    // ------------------------------------------------------------------------
    // Properties

    private static final String         PROP_digitalInputIndex  = "digitalInputIndex";  // [ignition|#]
    private static final String         PROP_reportType         = "reportType";         // detail|summary
    
    private static final String         REPORT_TYPE_detail      = "detail";
    private static final String         REPORT_TYPE_summary     = "summary";
    
    private static final boolean        FIXUP_FIRST_DIGITIAL_ON = false;

    // ------------------------------------------------------------------------
    // ------------------------------------------------------------------------
    
    private I18N                        i18n                    = null;
    
    private Device                      device                  = null;

    private String                      description             = null;

    private EventData                   firstEvent              = null;
    private EventData                   lastEvent               = null;

    private Map<Integer,AccumulatorLong> codeCount              = new OrderedMap<Integer,AccumulatorLong>();

    private Vector<FieldData>           rowData                 = null;

    // ------------------------------------------------------------------------

    /**
    *** DigitalInput Report Constructor
    *** @param rptEntry The ReportEntry that generated this report
    *** @param reqState The session RequestProperties instance
    *** @param devList  The list of devices
    **/
    public DailySummaryReport(ReportEntry rptEntry, RequestProperties reqState, ReportDeviceList devList)
        throws ReportException
    {
        super(rptEntry, reqState, devList);
        this.i18n = reqState.getPrivateLabel().getI18N(DailySummaryReport.class);

        /* Account check */
        if (this.getAccount() == null) {
            throw new ReportException("Account-ID not specified");
        }

        /* Device check */
        if (this.getDeviceCount() <= 0) {
            throw new ReportException("No Devices specified");
        }

    }

    // ------------------------------------------------------------------------

    /**
    *** Post report initialization
    **/
    public void postInitialize()
    {
        //ReportConstraints rc = this.getReportConstraints();
        //Print.logInfo("LimitType=" + rc.getSelectionLimitType() + ", Limit=" + rc.getSelectionLimit());
    }

    // ------------------------------------------------------------------------

    /**
    *** Returns true if this report handles only a single device at a time
    *** @return True If this report handles only a single device at a time
    **/
    public boolean isSingleDeviceOnly()
    {
        return false;
    }

    // ------------------------------------------------------------------------

    /**
    *** Override 'getEventData' to reset selected status codes
    *** @param device       The Device for which EventData records will be selected
    *** @param rcdHandler   The DBRecordHandler
    *** @return An array of EventData records for the device
    **/
    protected EventData[] getEventData(Device device, DBRecordHandler<EventData> rcdHandler)
    {

        /* Device */
        if (device == null) {
            return EventData.EMPTY_ARRAY;
        }

        /* adjust report constraints */
        ReportConstraints rc = this.getReportConstraints();
        rc.setValidGPSRequired(false); // don't need just valid gps events

        /* get events */
        return super.getEventData(device, rcdHandler);
        
    }

    // ------------------------------------------------------------------------

    /**
    *** Returns true if this report supports displaying a map
    *** @return True if this report supports displaying a map, false otherwise
    **/
    public boolean getSupportsMapDisplay()
    {
        return false;
    }

    // ------------------------------------------------------------------------

    /**
    *** Gets the bound ReportLayout singleton instance for this report
    *** @return The bound ReportLayout
    **/
    public static ReportLayout GetReportLayout()
    {
        // bind the report format to this data
        return FieldLayout.getReportLayout();
    }

    /**
    *** Gets the bound ReportLayout singleton instance for this report
    *** @return The bound ReportLayout
    **/
    public ReportLayout getReportLayout()
    {
        // bind the report format to this data
        return GetReportLayout();
    }

    // ------------------------------------------------------------------------

    /**
    *** Creates and returns an iterator for the row data displayed in the body of this report.
    *** @return The body row data iterator
    **/
    public DBDataIterator getBodyDataIterator()
    {

        /* init */
        this.rowData = new Vector<FieldData>();

        /* loop through devices (should be only 1) */
        String accountID = this.getAccountID();
        String devID = "";
        ReportDeviceList devList = this.getReportDeviceList();
        for (Iterator i = devList.iterator(); i.hasNext();) {
            devID = (String)i.next();
            try {

                /* get device */
                this.device = devList.getDevice(devID);

                /* reset */
                this.firstEvent = null;
                this.lastEvent  = null;
                this.codeCount.clear();

                /* get events */
                this.getEventData(this.device, this); // <== callback to 'handleDBRecord'

                /* last event */
                double odomDeltaKM = 0.0;
                double fuelDeltaL  = 0.0;
                if (this.lastEvent != null) {
                    odomDeltaKM = this.lastEvent.getOdometerKM() - this.firstEvent.getOdometerKM();
                    fuelDeltaL  = this.lastEvent.getFuelTotal()  - this.firstEvent.getFuelTotal();
                }

                /* create report line entry */
                FieldData fd = new FieldData();
                fd.setAccount(this.getAccount());
                fd.setString(FieldLayout.DATA_ACCOUNT_ID    , this.getAccountID());
                fd.setDevice(this.device);
                fd.setString(FieldLayout.DATA_DEVICE_ID     , devID);
                fd.setDouble(FieldLayout.DATA_DISTANCE      , odomDeltaKM);
                fd.setDouble(FieldLayout.DATA_ODOMETER_DELTA, odomDeltaKM);
                fd.setDouble(FieldLayout.DATA_FUEL_TOTAL    , fuelDeltaL);
                fd.setDouble(FieldLayout.DATA_FUEL_TRIP     , fuelDeltaL);
                for (Integer sc : this.codeCount.keySet()) {
                    int code = sc.intValue();
                    AccumulatorLong al = this.codeCount.get(sc);
                    String cKey = FieldLayout.DATA_STATUS_COUNT + "_" + StringTools.toHexString(code,16);
                    fd.setLong(cKey, al.get());
                }
                this.rowData.add(fd);

            } catch (DBException dbe) {
                Print.logError("Error retrieving EventData for Device: " + devID);
            }

        } // loop through devices

        /* return row iterator */
        // TODO: sort by deviceID
        return new ListDataIterator(this.rowData);
        
    }

    /**
    *** Creates and returns an iterator for the row data displayed in the total rows of this report.
    *** @return The total row data iterator
    **/
    public DBDataIterator getTotalsDataIterator()
    {

        /* return row iterator */
        return null;

    }

    // ------------------------------------------------------------------------
    // ------------------------------------------------------------------------
    
    /**
    *** Custom DBRecord callback handler class
    *** @param rcd  The EventData record
    *** @return The returned status indicating whether to continue, or stop
    **/
    public int handleDBRecord(EventData rcd)
        throws DBException
    {
        EventData ev = rcd;

        /* first event */
        if (this.firstEvent == null) {
            //Print.logInfo("First Event at " + ev.getTimestamp());
            this.firstEvent = ev;
        }

        /* count status codes */
        Integer sci = new Integer(ev.getStatusCode());
        AccumulatorLong acc = this.codeCount.get(sci);
        if (acc != null) {
            acc.increment();
        } else {
            this.codeCount.put(sci, new AccumulatorLong(1L));
        }

        /* last event */
        //Print.logInfo("Saving Last Event at " + ev.getTimestamp());
        this.lastEvent = ev;

        /* return record limit status */
        return (this.rowData.size() < this.getReportLimit())? DBRH_SKIP : DBRH_STOP;

    }

    // ------------------------------------------------------------------------
    // ------------------------------------------------------------------------

    protected long getReportStartTime()
    {
        long startTime = this.getTimeStart();
        return startTime;
    }

    protected long getReportEndTime()
    {
        long nowTime = DateTime.getCurrentTimeSec();
        long endTime = this.getTimeEnd();
        return (nowTime < endTime)? nowTime : endTime;
    }

    // ------------------------------------------------------------------------

}
